//------------------------------------------------------------------
//  classJson.ts
//  Copyright 2015 Applied Invention, LLC.
//------------------------------------------------------------------

import { Map } from '../util/Map';
import { MapKeyType } from '../util/Map';
import { ObjectMap } from '../util/types';

//------------------------------------------------------------------
// Types
//------------------------------------------------------------------

/** A ClassJson object after JSON encoding/decoding to a plain old ObjectMap.
 *
 * ClassJsonObject is a plain old ObjectMap, but also has a _class property.
 */
export interface ClassJsonObject
{
  _class: string;
  [key: string]: any;
}

//------------------------------------------------------------------
// Values
//------------------------------------------------------------------

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Writes an error message if a condition fails.
 *
 * Here's an example:
 *
 *   axe.classJson.assert(myStr == "ok",
 *                "Foo.bar",
 *                "I expected myStr to be 'ok', but it's actually '",
 *                myStr,
 *                "'.");
 *
 * @param condition The condition that must be true.
 * @param msg  Comma-separated pieces of an error message to be printed.
 *             This is a pseudo-varargs list.
 *             All these messages will be concatenated into an error message.
 */
export function assert(condition: boolean, ...msg: Array<any>) : void
{
  if (!condition) {

    let msgHeader = "ASSERT FAILED: ";

    let msgSegs: Array<string> = [];
    for (let i = 0; i < msg.length; ++i)
    {
      msgSegs.push("" + msg[i]);
    }
    msgSegs.unshift(msgHeader);

    // Write to the console to get pretty printed objects.
    if (typeof console != 'undefined')
    {
      console.error.apply(console, msgSegs);
    }

    throw new Error(msgSegs.join(" "));
  }
}

export interface Jsonable<T, U>
{
  fromJson(src: T) : U;
  toJson(src: U) : T;
}

/** MapKeyType for using classes as keys of a map.
 *
 * This assumes that all classes have a unique name.
 */
export class JsonableMapType implements MapKeyType<Jsonable<any, any>>
{
  typeName: string;

  constructor()
  {
    this.typeName = "Jsonable";
  }

  encode(key: Jsonable<any, any>) : string
  {
    // Use the function name.
    return (<any>key).name;
  }
}

/** Creates a map of classes to class names.
 */
export function createJsonableMap() : Map<Jsonable<any, any>, string>
{
  return new Map<Jsonable<any, any>, string>(new JsonableMapType());
}

export class JsonFactory
{
  map: Map<string, any> =
    Map.createStringMap<any>();

  register<T, U>(className: string, classFactory: Jsonable<T, U>) : void
  {
    this.map.put(className, classFactory);
  }

  create(className: string, jsonObject: ObjectMap<any>) : any
  {
    let ctor = this.map.get(className);

    return new ctor(42, "my name");
  }
}
