//------------------------------------------------------------------
//  JsonDate.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
/// <reference path="../../util/date" />
import * as axeClasses from "../../util/classes";
import * as axeDate from "../../util/date";
import * as axeString from "../../util/string";
/// <reference path="JsonType" />
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
import { Day } from '../../date/Day';
//------------------------------------------------------------------

/** Marks a class attribute as a date.
 *
 * This is a Day object in Typescript and an ISO 8601 string in JSON.
 */
export class JsonDate extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The format that dates shall have.
   */
  static DATE_REG_EXP: RegExp = new RegExp('\\d{4}-\\d{2}-\\d{2}');

  /** The format that dates shall have.
   */
  static DATE_PATTERN: string = 'YYYY-MM-DD';

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonDate object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (!(!value || value instanceof Day))
    {
      let msg = ('is of type date, ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (!value)
    {
      return null;
    }
    else if (typeof value != 'string')
    {
      let msg = ('is of type date (string), ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else if (!this.canBeParsed(value))
    {
      let msg = ('is of type date, and the value is not valid date format "' +
                 JsonDate.DATE_PATTERN + '"' +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Returns True if the specified string can be parsed as a date.
   */
  private canBeParsed(dateStr: string) : boolean
  {
    return JsonDate.DATE_REG_EXP.test(dateStr);
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }
    else
    {
      let valueObj = <Day>value;
      return axeDate.formatDate(valueObj);
    }
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }
    else
    {
      let valueStr = <string>value;
      return axeDate.parseDate(valueStr);
    }
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    return value;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("JsonDate", this, propertyNames);
  }

} // END class JsonDate
