//------------------------------------------------------------------
//  TestClassJsonEncoder.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../util/string";
import { ObjectMap } from "../../util/types";
import { Map } from "../../util/Map";
import { ClassJsonEncoder } from '../ClassJsonEncoder';
import { ClassJsonDesc } from '../ClassJsonDesc';
import { ClassJsonRegistry } from '../ClassJsonRegistry';
import { Day } from "../../date/Day";
import { Duration } from "../../date/Duration";
import { JsonDate } from "../jsonTypes/JsonDate";
import { JsonDateTime } from "../jsonTypes/JsonDateTime";
import { JsonDuration } from "../jsonTypes/JsonDuration";
import { JsonLink } from "../jsonTypes/JsonLink";
import { JsonList } from "../jsonTypes/JsonList";
import { JsonMap } from "../jsonTypes/JsonMap";
import { JsonObj } from "../jsonTypes/JsonObj";
import { JsonPrimitiveType } from "../jsonTypes/JsonPrimitiveType";
import { UnitTest } from '../../unittest/UnitTest';
import { UnitTestRunner } from '../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the ClassJsonEncoder class.
 */
export class TestClassJsonEncoder extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonEncoder object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods
  //------------------------------------------------------------------

  /** Test encoding psuedo-primitive types.
   */
  testEncodeBuiltins() : void
  {
    interface IDateHolder
    {
      date: Day;
      datetime: Date;
      duration: Duration;
      dateList: Array<Day>;
      datetimeList: Array<Date>;
      durationList: Array<Duration>;
      dateMap: Map<string, Day>;
      datetimeMap: Map<string, Date>;
      durationMap: Map<string, Duration>;
    }

    class DateHolder
    {
      dateMap: Map<string, Day> = Map.createStringMap<Day>();
      datetimeMap: Map<string, Date> = Map.createStringMap<Date>();
      durationMap: Map<string, Duration> = Map.createStringMap<Duration>();

      /* tslint:disable:no-parameter-properties */
      constructor(public date: Day,
                  public datetime: Date,
                  public duration: Duration,
                  public dateList: Array<Day>,
                  public datetimeList: Array<Date>,
                  public durationList: Array<Duration>)
      {
      }

      static fromJson(src: IDateHolder) : DateHolder
      {
        let obj = new DateHolder(src.date, src.datetime, src.duration,
                                 src.dateList, src.datetimeList,
                                 src.durationList);
        obj.dateMap.putAll(src.dateMap);
        obj.datetimeMap.putAll(src.datetimeMap);
        obj.durationMap.putAll(src.durationMap);

        return obj;
      }

      static toJson(src: DateHolder) : IDateHolder
      {
        return src;
      }
    }

    let dateHolderDesc = new ClassJsonDesc("DateHolder");

    dateHolderDesc.addField("date", new JsonDate());
    dateHolderDesc.addField("datetime", new JsonDateTime());
    dateHolderDesc.addField("duration", new JsonDuration());
    dateHolderDesc.addField("dateList", new JsonList(new JsonDate()));
    dateHolderDesc.addField("datetimeList", new JsonList(new JsonDateTime()));
    dateHolderDesc.addField("durationList", new JsonList(new JsonDuration()));
    dateHolderDesc.addField("dateMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDate()));
    dateHolderDesc.addField("datetimeMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDateTime()));
    dateHolderDesc.addField("durationMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDuration()));

    ClassJsonRegistry.registry.addDesc(dateHolderDesc);
    ClassJsonRegistry.registry.register("DateHolder", DateHolder);

    // 2016-01-02
    let theDate = new Day(2016, 0, 2);

    // 2016-01-04T01:02:03.456Z
    let theDatetime = new Date(Date.UTC(2016, 0, 4, 1, 2, 3, 456));

    let theDuration = new Duration(1000);

    let dateHolder = new DateHolder(theDate,
                                    theDatetime,
                                    theDuration,
                                    [theDate],
                                    [theDatetime],
                                    [theDuration]);
    dateHolder.dateMap.put("date", theDate);
    dateHolder.datetimeMap.put("datetime", theDatetime);
    dateHolder.durationMap.put("duration", theDuration);

    let expected: string = `{
      "_class": "DateHolder",
      "date": "2016-01-02",
      "datetime": "2016-01-04T01:02:03.456Z",
      "duration": 1000,
      "dateList": ["2016-01-02"],
      "datetimeList": ["2016-01-04T01:02:03.456Z"],
      "durationList": [1000],
      "dateMap" : [
        ["date", "2016-01-02"]
      ],
      "datetimeMap" : [
        ["datetime", "2016-01-04T01:02:03.456Z"]
      ],
      "durationMap" : [
        ["duration", 1000]
      ]
    }`;

    // Strip out all whitespace.
    expected = this.compressJson(expected);

    let actual = (new ClassJsonEncoder()).encode(dateHolder);

    this.assertEqual("dates", expected, actual);
  }

  /** Test classes.
   */
  testEncodeNestedClasses() : void
  {
    let wheelDesc = new ClassJsonDesc("Wheel");
    wheelDesc.addField("diameter", new JsonPrimitiveType("int"));
    wheelDesc.addField("pressure", new JsonPrimitiveType("float"));

    let carDesc = new ClassJsonDesc("Car");
    carDesc.addField("color", new JsonPrimitiveType("string"));
    carDesc.addField("wheels", new JsonList(new JsonObj("Wheel")));
    carDesc.addField("namedWheels",
                     new JsonMap(new JsonPrimitiveType("string"),
                                 new JsonObj("Wheel")));

    ClassJsonRegistry.registry.addDesc(wheelDesc);
    ClassJsonRegistry.registry.addDesc(carDesc);

    interface ICar
    {
      color: string;
      wheels: Array<Wheel>;
      namedWheels: Map<string, Wheel>;
    }

    class Car implements ICar
    {
      namedWheels: Map<string, Wheel> = Map.createStringMap<Wheel>();

      constructor(public color: string, public wheels: Array<Wheel>)
      {
      }

      static fromJson(src: ICar) : Car
      {
        return new Car(src.color, src.wheels);
      }

      static toJson(src: Car) : ICar
      {
        return src;
      }
    }

    interface IWheel
    {
      diameter: number;
      pressure: number;
    }

    class Wheel implements IWheel
    {
      constructor(public diameter: number, public pressure: number)
      {
      }

      static fromJson(src: IWheel) : Wheel
      {
        return new Wheel(src.diameter, src.pressure);
      }

      static toJson(src: Wheel) : IWheel
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register("Car", Car);
    ClassJsonRegistry.registry.register("Wheel", Wheel);

    let car: Car = new Car("red", [new Wheel(5, 3.16), new Wheel(6, 3.17)]);

    car.namedWheels.put("front", new Wheel(3, 3.14));
    car.namedWheels.put("back", new Wheel(4, 3.15));

    let jsonStr: string = `{
      "_class" : "Car",
      "color" : "red",
      "wheels" : [
        {"_class": "Wheel", "diameter": 5, "pressure": 3.16},
        {"_class": "Wheel", "diameter": 6, "pressure": 3.17}
      ],
      "namedWheels" : [
        ["front", {"_class": "Wheel", "diameter": 3, "pressure": 3.14}],
        ["back", {"_class": "Wheel", "diameter": 4, "pressure": 3.15}]
      ]
    }`;

    jsonStr = this.compressJson(jsonStr);

    let actual = (new ClassJsonEncoder()).encode(car);

    this.assertEqual("objects", jsonStr, actual);
  }

  /** An unregistered class should throw an exception.
   */
  testUnregisteredClass() : void
  {
    try
    {
      (new ClassJsonEncoder()).encode(new ClassJsonEncoder());
      this.fail("Failed to throw unregistered class exception");
    }
    catch (ex)
    {
    }
  }

  /** A missing property should throw an exception.
   */
  testMissingProperty() : void
  {
    let desc = new ClassJsonDesc("NamedPoint");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint
    {
      constructor(public name: string, public x: number, public y: number)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    let obj = new NamedPoint('name', 33, 34);
    delete obj.x;

    try
    {
      (new ClassJsonEncoder()).encode(obj);
      this.fail("Failed to throw undecorated class exception");
    }
    catch (ex)
    {
    }
  }

  /** A mis-typed property should throw an exception.
   */
  testWrongType() : void
  {
    let desc = new ClassJsonDesc("NamedPoint");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint
    {
      constructor(public name: string, public x: number, public y: number)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    let obj: NamedPoint = new NamedPoint('name', 33, 34);

    // Make y a string instead of an int.
    // (The <any> cast is required to circumvent type-checking.)
    (<any>obj).y = 'foo';

    try
    {
      (new ClassJsonEncoder()).encode(obj);
      this.fail("Failed to throw wrong property type exception");
    }
    catch (ex)
    {
    }
  }

  /** Test JSON encoding of nested dictionaries.
   */
  testNestedDictEncode() : void
  {
    let desc = new ClassJsonDesc("DictHolder");
    desc.addField("theDict",
                  new JsonMap(new JsonPrimitiveType("int"),
                              new JsonMap(new JsonPrimitiveType("int"),
                                          new JsonPrimitiveType("bool"))));
    ClassJsonRegistry.registry.addDesc(desc);

    class DictHolder
    {
      constructor(public theDict: Map<number, Map<number, boolean>>)
      {
      }

      static fromJson(src: ObjectMap<any>) : DictHolder
      {
        let parsedDict = <Map<number, Map<number, boolean>>>src['theDict'];
        return new DictHolder(parsedDict);
      }

      static toJson(src: DictHolder) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, DictHolder);


    let subDict1 = Map.createIntMap<boolean>();
    let subDict2 = Map.createIntMap<boolean>();
    let rootDict = Map.createIntMap<Map<number, boolean>>();

    subDict1.putPairs([[1, true], [2, true], [3, false]]);
    subDict2.putPairs([[4, true], [5, true], [6, false]]);
    rootDict.putPairs([[100, subDict1], [101, subDict2]]);
    let obj = new DictHolder(rootDict);

    let encodedSubDict1 = '[[1, true], [2, true], [3, false]]';
    let encodedSubDict2 = '[[4, true], [5, true], [6, false]]';
    let encodedDict = ('[[100, ' + encodedSubDict1 + '], ' +
                       '[101, ' + encodedSubDict2 + ']]');
    let expected = ('{"_class": "DictHolder", ' +
                    '"theDict": ' + encodedDict + '}');
    expected = this.compressJson(expected);

    let actual = (new ClassJsonEncoder()).encode(obj);

    this.assertEqual('nested dict', expected, actual);
  }

  /** Test JSON encoding of nested lists.
   */
  testNestedListEncode() : void
  {
    let intType = new JsonPrimitiveType("int");

    let desc = new ClassJsonDesc("ListHolder");
    desc.addField("theList",
                  new JsonList(new JsonList(new JsonList(intType))));
    ClassJsonRegistry.registry.addDesc(desc);

    class ListHolder
    {
      constructor(public theList: Array<Array<Array<number>>>)
      {
      }

      static fromJson(src: ObjectMap<any>) : ListHolder
      {
        return new ListHolder(<Array<Array<Array<number>>>>src['theList']);
      }

      static toJson(src: ListHolder) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, ListHolder);


    let heldList = [
      [[0, 1], [0, 2], [0, 3]],
      [[1, 1], [1, 2], [1, 3]],
    ];

    let obj = new ListHolder(heldList);

    let actual = (new ClassJsonEncoder()).encode(obj);

    let actualList = JSON.parse(actual)['theList'];

    this.assertEqual('0, 0', [0, 1], actualList[0][0]);
    this.assertEqual('1, 0', [1, 1], actualList[1][0]);
    this.assertEqual('1, 2', [1, 3], actualList[1][2]);
  }

  /** Test JSON encoding of a subclass.
   */
  testSubclass() : void
  {
    // A base class (NamedThing) and two subclasses (NamedPoint, NamedAmount).

    let desc = new ClassJsonDesc("NamedThing");
    desc.setAbstract(true);
    desc.addField("name", new JsonPrimitiveType("string"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedThing
    {
      constructor(public name: string)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedThing
      {
        return new NamedThing(<string>src['name']);
      }

      static toJson(src: NamedThing) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedThing);


    desc = new ClassJsonDesc("NamedPoint");
    desc.setBaseClass("NamedThing");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint extends NamedThing
    {
      constructor(name: string, public x: number, public y: number)
      {
        super(name);
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    desc = new ClassJsonDesc("NamedAmount");
    desc.setBaseClass("NamedThing");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("amount", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedAmount extends NamedThing
    {
      constructor(name: string, public amount: number)
      {
        super(name);
      }

      static fromJson(src: ObjectMap<any>) : NamedAmount
      {
        return new NamedAmount(<string>src['name'],
                               <number>src['amount']);
      }

      static toJson(src: NamedAmount) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedAmount);


    let encoder = new ClassJsonEncoder();

    let obj: NamedThing = new NamedPoint("hello", 3, 7);
    let expected = '{"_class": "NamedPoint", "name": "hello", "x": 3, "y": 7}';
    expected = this.compressJson(expected);

    let actual = encoder.encode(obj);

    this.assertEqual('NamedPoint', expected, actual);

    obj = new NamedAmount("hello", 42);
    expected = '{"_class": "NamedAmount", "name": "hello", "amount": 42}';
    expected = this.compressJson(expected);

    actual = encoder.encode(obj);

    this.assertEqual('NamedAmount', expected, actual);

    obj = new NamedThing("hello");

    try
    {
      actual = encoder.encode(obj);
      this.fail("No exception for encoding abstract class.");
    }
    catch (ex)
    {
      this.assertEqual("abstract message",
                       true,
                       ex.message.indexOf("abstract class") > 0);
    }
  }

  /** Test that links are handled correctly.
   */
  testLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonList(new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstument")));

    class TestClassBand
    {
      musicians: Array<TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Array<TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Array<TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);

    let instruments = [
      new TestClassInstrument(20),
      new TestClassInstrument(21),
      new TestClassInstrument(22),
    ];

    let musicians = [
      new TestClassMusician(10, instruments[2]),
      new TestClassMusician(11, instruments[0]),
    ];

    let band = new TestClassBand(musicians, instruments);


    let bandJsonStr = (new ClassJsonEncoder()).encode(band);
    let bandJson = JSON.parse(bandJsonStr);
    let musiciansJson: Array<ObjectMap<any>> = bandJson['musicians'];

    let musicianJson = musiciansJson[0];
    let instrumentJson = musicianJson['instrument'];

    this.assertEqual('instrument link is the ID.', [22], instrumentJson);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

  /** Compresses JSON to the non-pretty version we expect.
   */
  private compressJson(text: string) : string
  {
    return JSON.stringify(JSON.parse(text));
  }

} // END class TestClassJsonEncoder

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestClassJsonEncoder());
