//------------------------------------------------------------------
//  UnitTestRunner.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
/// <reference path="../util/func" />
import * as axeClasses from "../util/classes";
import * as axeString from "../util/string";
import { UnitTest } from './UnitTest';
import { Map } from '../util/Map';
//------------------------------------------------------------------

/** Runs a group of UnitTest objects.
 */
export class UnitTestRunner
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** All created tests.
   */
  private static allTests: Array<UnitTest> = [];

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new UnitTestRunner object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Static method registers a test.
   */
  static add(test: UnitTest) : void
  {
    UnitTestRunner.allTests.push(test);
  }

  /** Static method runs a single registered tests.
   *
   * @return An error message listing all failed tests.
   *         Will be null if all tests passed.
   */
  static runTest(name: string) : string
  {
    for (let testObj of UnitTestRunner.allTests)
    {
      let className: string = axeClasses.className(testObj);

      if (className == axeString.capitalize(name) ||
          className == 'Test' + axeString.capitalize(name))
      {
        return testObj.run();
      }
    }
  }

  /** Static method runs all registered tests.
   *
   * @return A map of all failed tests.
   *         The map's keys are class names, and values are
   *         error messages listing all failed tests.
   *         The map will be empty if all tests passed.
   */
  static runAll() : Map<string, string>
  {
    let ret: Map<string, string> = Map.createStringMap<string>();

    for (let testObj of UnitTestRunner.allTests)
    {
      let className: string = testObj.constructor.name;
      className = className.substring('Test'.length);

      let statusMsg = testObj.run();

      if (statusMsg)
      {
        ret.put(className, statusMsg);
      }
    }

    return ret;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("UnitTestRunner", this, propertyNames);
  }

} // END class UnitTestRunner
