//------------------------------------------------------------------
//  Point.ts
//  Copyright 2014 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
//------------------------------------------------------------------

/** An XyPoint is an object that has an 'x' and 'y' property.
 */
export interface XyPoint
{
  x: number;
  y: number;
}

/** An (x, y) point.
 */
export class Point implements XyPoint
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The x-value of the point.
   */
  public x: number;

  /** The y-value of the point.
   */
  public y: number;

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Creates a new Point object.
   */
  constructor(x: number, y: number)
  {
    this.x = x;
    this.y = y;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Copies the x/y from another point to this one.
   */
  copyFrom(point: XyPoint) : void
  {
    this.x = point.x;
    this.y = point.y;
  }

  /** Sets the x, y of this point.
   */
  setXy(x: number, y: number) : void
  {
    this.x = x;
    this.y = y;
  }

  /** Sets the x, y of this point.
   */
  setXyArray(xyArray: Array<number>) : void
  {
    this.x = xyArray[0];
    this.y = xyArray[1];
  }

  /** Returns the distance between this point and another.
   */
  distanceFrom(point: XyPoint) : number
  {
    let x = this.x;
    let y = this.y;

    return Math.sqrt(Math.pow(point.x - x, 2) + Math.pow(point.y - y, 2));
  }

  /** Returns an array [x, y] representation of this object.
   */
  toArray() : Array<number>
  {
    return [this.x, this.y];
  }

  /** Converts from a JSON object to this class.
   */
  /*
  static fromJson(src: axe.cjs.Point) : Point
  {
    return new Point(src.x, src.y);
  }
  */

  /** Converts from this class to a JSON-ready object.
   */
  /*
  static toJson(src: Point) : axe.cjs.Point
  {
    return src;
  }
  */

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames = [ 'x', 'y' ];
    return axeString.formatObject("Point", this, propertyNames);
  }

} // END class Point
