//------------------------------------------------------------------
//  TestJsonLink.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../../../util/string';
import { ClassJsonDesc } from '../../ClassJsonDesc';
import { ClassJsonRegistry } from '../../ClassJsonRegistry';
import { Jsonable } from '../../classJson';
import { JsonLink } from '../JsonLink';
import { JsonList } from '../JsonList';
import { JsonObj } from '../JsonObj';
import { JsonPrimitiveType } from '../JsonPrimitiveType';
import { AnyCtor } from '../../../util/types';
import { ObjectMap } from '../../../util/types';
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the JsonLink class.
 */
export class TestJsonLink extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonLink object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** Test the validateJson() method.
   */
  testValidateJson() : void
  {
    let [testClass, jsonType] = this.createClassType();

    let testObj = new testClass(42, 43);

    this.assertNull('object', jsonType.validate(testObj));
    this.assertNull('int', jsonType.validateJson([42]));

    this.assertNull('null', jsonType.validate(null));
    this.assertNull('null', jsonType.validateJson(null));
  }

  /** An object that has no ID field.
   */
  testObjectHasNoId() : void
  {
    let [testClass, jsonType] = this.createClassType();

    let testObj = new testClass(42, 43);

    // Remove this object's 'id' member.
    delete testObj['id'];

    this.assertNotNull('object with no ID', jsonType.validate(testObj));
  }

  /** An object that is the wrong class.
   */
  testWrongClass() : void
  {
    let [testClass, jsonType] = this.createClassType();

    // Should be a TestClass, but is actually an int.
    let wrongClassObj: any = 43;
    this.assertNotNull('wrong class', jsonType.validate(wrongClassObj));


    // Should be an int, but is actually a TestClass.
    wrongClassObj = new testClass(42, 43);
    this.assertNotNull('wrong class', jsonType.validateJson(wrongClassObj));
  }

  /** Test the encode() method.
   */
  testEncode() : void
  {
    let [testClass, jsonType] = this.createClassType();
    let testObj = new testClass(42, 43);

    this.assertEqual('encode', [42], jsonType.encode(testObj));
  }

  /** Test the decode() method.
   */
  testDecode() : void
  {
    let [testClass, jsonType] = this.createClassType();
    let testObj = new testClass(42, 43);

    this.assertEqual('decode', [3], jsonType.decode([3]));
    this.assertEqual('decode null', null, jsonType.decode(null));
  }

  /** Test the decodeLinks() method.
   */
  testDecodeLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonList(new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstument")));

    class TestClassBand
    {
      musicians: Array<TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Array<TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Array<TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);

    let instruments = [
      new TestClassInstrument(20),
      new TestClassInstrument(21),
      new TestClassInstrument(22),
    ];

    let musicians = [
      new TestClassMusician(10, instruments[2]),
      new TestClassMusician(11, instruments[0]),
    ];

    let band = new TestClassBand(musicians, instruments);

    // Link are unresolved, so instead of pointers to objects,
    // they are the object IDs.

    for (let musician of musicians)
    {
      (<any>musician.instrument) = [musician.instrument.id];
    }

    let jsonType =
      new JsonLink('TestClassInstrument', '../instruments', ['id']);
    let parents: Array<object> = [band, musicians[0]];

    this.assertNull('null', jsonType.decodeLinks(parents, null));

    let linkedObj = jsonType.decodeLinks(parents, musicians[0].instrument);

    this.assertEqual('musician 0', instruments[2], linkedObj);


    // Invalid state: link target is not a list:

    let jsonType2 =
      new JsonLink('TestClassInstrument', '..', ['id']);

    try
    {
      jsonType2.decodeLinks(parents, [musicians[0].instrument]);
      this.fail("No exception for non-list link target.");
    }
    catch
    {
    }


    // Invalid state: link target item classes.

    desc = new ClassJsonDesc("TestClassFoo");
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassFoo
    {
      fooAttr: number;
      constructor(fooAttr: number)
      {
        this.fooAttr = fooAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClassFoo
      {
        return new TestClassFoo(<number>src['fooAttr']);
      }

      static toJson(src: TestClassFoo) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassFoo);

    let jsonType3 = new JsonLink('TestClassFoo', '../instruments', ['fooAttr']);

    try
    {
      jsonType3.decodeLinks(parents, musicians[0].instrument);
      this.fail("No exception for wrong list item type.");
    }
    catch
    {
    }


    // Invalid state: link ID not found.

    let badMusician = new TestClassMusician(12, instruments[0]);
    (<any>badMusician.instrument) = [-1];

    try
    {
      console.log("BEGIN CHECK");
      jsonType.decodeLinks(parents, badMusician.instrument);
      console.log("END CHECK");
      this.fail("No exception for link ID not found.");
    }
    catch
    {
    }
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

  /** Private helper metod to create a type.
   */
  createType() : JsonLink
  {
    return this.createClassType()[1];
  }

  /** Private helper metod to create a type.
   */
  createClassType() : [AnyCtor, JsonLink]
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      id: number;
      fooAttr: number;

      constructor(id: number, fooAttr: number)
      {
        this.id = id;
        this.fooAttr = fooAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['id'],
                             <number>src['fooAttr']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);

    let jsonType = new JsonLink('TestClass', 'a/b/c', ['id']);

    return [TestClass, jsonType];
  }

} // END class TestJsonLink

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonLink());
