//------------------------------------------------------------------
//  TestJsonTuple.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../../util/string";
import { ClassJsonDesc } from '../../ClassJsonDesc';
import { ClassJsonRegistry } from '../../ClassJsonRegistry';
import { JsonLink } from '../JsonLink';
import { JsonList } from '../JsonList';
import { JsonTuple } from '../JsonTuple';
import { JsonObj } from '../JsonObj';
import { JsonPrimitiveType } from '../JsonPrimitiveType';
import { ObjectMap } from "../../../util/types";
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the JsonTuple class.
 */
export class TestJsonTuple extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonTuple object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** A wrong tuple type should throw an exception.
   */
  testBadTupleType() : void
  {
    let jsonType = new JsonTuple([new JsonPrimitiveType('string'),
                                  new JsonPrimitiveType('int')]);

    let errorMsg = jsonType.validate('bad value');
    this.assertNotNull('bad value', errorMsg);

    errorMsg = jsonType.validateJson('bad value');
    this.assertNotNull('bad value json', errorMsg);
  }

  /** A wrong tuple length should throw an exception.
   */
  testBadTupleLength() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['foo']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    desc = new ClassJsonDesc("TestClass2");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass2
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass2
      {
        return new TestClass2(<number>src['foo']);
      }

      static toJson(src: TestClass2) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass2);


    let jsonType = new JsonTuple([new JsonPrimitiveType('int'),
                                  new JsonObj('TestClass')]);

    let errorMsg = jsonType.validate([new TestClass(1),
                                      new TestClass2(2),
                                      'bad value 2']);
    this.assertNotNull('bad tuple length', errorMsg);

    let jsonTestClass = {'_class': 'TestClass', 'foo': 1};
    let jsonTestClass2 = {'_class': 'TestClass2', 'foo': 2};

    let jsonTuple = [jsonTestClass, jsonTestClass2, 'bad value 2'];
    errorMsg = jsonType.validateJson(jsonTuple);

    this.assertNotNull('bad tuple json length', errorMsg);
  }

  /** A wrong tuple item type should throw an exception.
   */
  testBadTupleItemType() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['foo']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    desc = new ClassJsonDesc("TestClass2");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass2
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass2
      {
        return new TestClass2(<number>src['foo']);
      }

      static toJson(src: TestClass2) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass2);


    let jsonType = new JsonTuple([new JsonPrimitiveType('int'),
                                  new JsonObj('TestClass')]);

    let errorMsg = jsonType.validate([42,
                                      new TestClass2(2)]);
    this.assertNotNull('bad tuple types', errorMsg);

    let jsonTestClass2 = {'_class': 'TestClass2', 'foo': 2};

    let jsonTuple = [42, jsonTestClass2];
    errorMsg = jsonType.validateJson(jsonTuple);

    this.assertNotNull('bad tuple json types', errorMsg);
  }

  /** Test the validateJson() method.
   */
  testValidateJson() : void
  {
    let jsonType = new JsonTuple([new JsonPrimitiveType('string'),
                                  new JsonPrimitiveType('int')]);

    this.assertNull('validate', jsonType.validate(['foo', 42]));
    this.assertNull('validateJson', jsonType.validateJson(['foo', 42]));

    this.assertNull('validate null', jsonType.validate(null));
    this.assertNull('validateJson null', jsonType.validateJson(null));
  }

  /** Test the encode() method.
   */
  testEncode() : void
  {
    let jsonType = new JsonTuple([new JsonPrimitiveType('string'),
                                  new JsonPrimitiveType('int')]);
    this.assertEqual('no change', ['foo', 3], jsonType.encode(['foo', 3]));
    this.assertEqual('no change null', null, jsonType.encode(null));
  }

  /** Test the decode() method.
   */
  testDecode() : void
  {
    let jsonType = new JsonTuple([new JsonPrimitiveType('string'),
                                  new JsonPrimitiveType('int')]);
    this.assertEqual('no change', ['foo', 3], jsonType.decode(['foo', 3]));
    this.assertEqual('no change null', null, jsonType.decode(null));
  }

  /** Test the decodeLinks() method.
   */
  testDecodeLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonList(new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstument")));

    class TestClassBand
    {
      musicians: Array<TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Array<TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Array<TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);

    let instruments = [
      new TestClassInstrument(20),
      new TestClassInstrument(21),
      new TestClassInstrument(22),
    ];

    let musicians = [
      new TestClassMusician(10, instruments[2]),
      new TestClassMusician(11, instruments[0]),
    ];

    let band = new TestClassBand(musicians, instruments);

    // Link are unresolved, so instead of pointers to objects,
    // they are the object IDs.

    for (let musician of musicians)
    {
      (<any>musician.instrument) = [musician.instrument.id];
    }

    let jsonType = new JsonTuple([new JsonPrimitiveType('int'),
                                  new JsonObj('TestClassMusician'),
                                  new JsonObj('TestClassMusician')]);
    let parents: Array<object> = [band];

    this.assertNull('null', jsonType.decodeLinks(parents, null));

    let dataTuple: [number, TestClassMusician, TestClassMusician];
    dataTuple = [42, musicians[0], musicians[1]];

    jsonType.decodeLinks(parents, dataTuple);

    this.assertEqual('int', 42, dataTuple[0]);
    this.assertEqual('musician 0', instruments[2], dataTuple[1].instrument);
    this.assertEqual('musician 1', instruments[0], dataTuple[2].instrument);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestJsonTuple

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonTuple());
