//------------------------------------------------------------------
//  JsonPath.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from '../../util/array';
import * as axeString from '../../util/string';
import { JsonPathElement } from './JsonPathElement';
//------------------------------------------------------------------

/** A path to a location in a JSON tree.
 */
export class JsonPath
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The elements that make up this path.
   */
  elements: Array<JsonPathElement>;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonPath object.
   */
  constructor(elements: Array<JsonPathElement>)
  {
    this.elements = elements;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Returns a JsonPath object for the specified string.
   *
   * The string is a slash-seperated series of names, each with
   * an optional selector in brackets.
   *
   * Some examples:
   *
   *   produced/bicycle/model
   *   produced/car/models[name]
   *   ../produced/car/models[name]
   */
  static parse(pathStr: string) : JsonPath
  {
    if (pathStr.startsWith('/'))
    {
      let msg = ('Error parsing link path.  The path should be relative.  ' +
                 'It should not start with a "/".  Path: ' + pathStr);
      throw Error(msg);
    }
    else
    {
      let elementStrs: Array<string> = pathStr.split('/');
      let elements: Array<JsonPathElement> = [];

      for (let elementStr of elementStrs)
      {
        elements.push(JsonPathElement.parse(elementStr, pathStr));
      }

      return new JsonPath(elements);
    }
  }

  /** Returns the object that this path point to.
   *
   * @param parents The list of ancestor objects of the object that we're
   *                starting to resove at.
   * @param relativeTo This path will be resolved relative to this object.
   */
  resolve(parents: Array<object>, relativeTo: object) : object
  {
    try
    {
      return this.resolveLevel(parents, relativeTo);
    }
    catch (ex)
    {
      let msg = "Error resolving path:  " + this.pathString();
      throw new Error(msg + '  ' + ex.toString());
    }
  }


  /** Resolves a single level of this path.
   *
   * @param parents The list of ancestor objects of the object that we're
   *                starting to resove at.
   * @param relativeTo This path will be resolved relative to this object.
   */
  resolveLevel(parents: Array<object>, relativeTo: object) : object
  {
    if (this.elements.length == 0)
    {
      return relativeTo;
    }

    let element: JsonPathElement = this.elements[0];
    let pathRemainder = new JsonPath(this.elements.slice(1));

    if (element.name == '..')
    {
      if (parents.length == 0)
      {
        let msg = ("Error resolving path:  Can't go up a level because there " +
                   "are no parents.  Remaining path:  "  + this.pathString());
        throw new Error(msg);
      }
      return pathRemainder.resolveLevel(parents.slice(0, -1),
                                        axeArray.last(parents));
    }
    else
    {
      let newParents: Array<object> = parents.concat([relativeTo]);
      let child: object = element.resolve(relativeTo);
      return pathRemainder.resolveLevel(newParents, child);
    }
  }


  /** Returns the path string for this path.
   */
  pathString() : string
  {
    let elementStrs: Array<string> = [];
    for (let element of this.elements)
    {
      elementStrs.push(element.pathString());
    }

    return elementStrs.join('/');
  }


  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("JsonPath", this, propertyNames);
  }

} // END class JsonPath
