//------------------------------------------------------------------
//  Angle.ts
//  Copyright 2014 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "./string";
import * as axeMath from './math';
import { ClassJsonObject } from '../classJson/classJson';
//------------------------------------------------------------------

/** A geometric angle.
 *
 * You can create an Angle from a number of degrees of radians
 * using static constructor functions like so:
 *
 *   var angle = Angle.degrees(180);
 *   var angle = Angle.radians(2 * Math.PI);
 *
 * Once created, you can access the value in degress or radians:
 *
 *    var numRadians = angle.radians;
 *    var numDegrees = angle.degrees;
 */
export class Angle
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** This angle in degrees.
   */
  public degrees: number;

  /** This angle in radians.
   */
  public radians: number;

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Initializes a new Angle object.
   */
  constructor(degrees: number, radians: number)
  {
    this.degrees = degrees;
    this.radians = radians;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Creates a new Angle object that is the specified number of degrees.
   *
   * @param degrees The number of degrees to create the new Angle with.
   *
   * @return A new Angle object.
   */
  static degrees(degrees: number) : Angle
  {
    degrees = Angle.normalizedDegrees(degrees);
    let radians: number = degrees * axeMath.radiansPerDegree;

    return new Angle(degrees, radians);
  }

  /** Creates a new Angle object that is the specified number of radians.
   *
   * @param radians The number of radians to create the new Angle with.
   *
   * @return A new Angle object.
   */
  static radians(radians: number) : Angle
  {
    radians = Angle.normalizedRadians(radians);
    let degrees: number = radians * axeMath.degreesPerRadian;

    return new Angle(degrees, radians);
  }

  /** Creates a new Angle that is the arctangent of the delta y and x.
   *
   * @param y The delta y.
   * @param x The delta x.
   */
  static atan2(y: number, x: number) : Angle
  {
    let radians = Math.atan2(y, x);
    return Angle.radians(radians);
  }

  /** Returns an equivalent number of degrees that is 0 <= x < 360.
   *
   * @param degrees The number of degrees to be normalized.
   *
   * @return The normalized number of degrees.
   */
  static normalizedDegrees(degrees: number) : number
  {
    return Angle.normalized(degrees, 360);
  }

  /** Returns an equivalent number of radians that is 0 <= x < 2 * PI.
   *
   * @param radians The number of radians to be normalized.
   *
   * @return The normalized number of radians.
   */
  static normalizedRadians(radians: number) : number
  {
    return Angle.normalized(radians, 2 * Math.PI);
  }

  /** Returns an equivalent number that is 0 <= x < maxAmount.
   *
   * @param n The number to be normalized.
   *
   * @return The normalized number.
   */
  static normalized(n: number, maxAmount: number) : number
  {
    n = n % maxAmount;
    if (n < 0)
    {
      n += maxAmount;
    }

    return n;
  }

  /** Adds this angle to another angle.
   *
   * @param otherAngle The other angle to be added.
   *
   * @return A new Angle object that is the sum of the two angles.
   */
  plus(otherAngle: Angle) : Angle
  {
    let newDegrees = Angle.normalizedDegrees(this.degrees + otherAngle.degrees);
    return Angle.degrees(newDegrees);
  }

  /** Subtracts another angle from this one.
   *
   * @param otherAngle The other angle to be subtracted.
   *
   * @return A new Angle object that is the difference of the two angles.
   */
  minus(otherAngle: Angle) : Angle
  {
    let newDegrees = Angle.normalizedDegrees(this.degrees - otherAngle.degrees);
    return Angle.degrees(newDegrees);
  }

  /** Multiplies this angle by a number n.
   *
   * @param n The number to multiply this angle by.
   *
   * @return A new Angle object that is the product of the two numbers.
   */
  times(n: number) : Angle
  {
    let newDegrees = Angle.normalizedDegrees(this.degrees * n);
    return Angle.degrees(newDegrees);
  }

  /** Divides this angle by a number n.
   *
   * @param n The number to divide this angle by.
   *
   * @return A new Angle object that is the qotient of the two numbers.
   */
  dividedBy(n: number) : Angle
  {
    //if (typeof n != "number")
    //{
    //  return this.dividedByAngle(n);
    //}

    let newDegrees = Angle.normalizedDegrees(this.degrees / n);
    return Angle.degrees(newDegrees);
  }

  /** Divides this angle by an Angle.
   *
   * @param angle The angle to divide this angle by.
   *
   * @return A number that is the quotient of the two angles.
   */
  dividedByAngle(angle: Angle) : number
  {
    return this.radians / angle.radians;
  }

  /** Returns the cosine of this angle.
   */
  cos() : number
  {
    return Math.cos(this.radians);
  }

  /** Returns the sine of this angle.
   */
  sin() : number
  {
    return Math.sin(this.radians);
  }

  /** Returns the tangent of this angle.
   */
  tan() : number
  {
    return Math.tan(this.radians);
  }

  /** Converts this 'math' angle to a 'map' angle.
   *
   * A 'math' angle starts at 3 o'clock and increases counterclockwise.
   * A 'map' angle starts at 12 o'clock and increases clockwise.
   *
   * @return A new Angle object which is the equivalent 'map' angle.
   */
  mathToMapAngle() : Angle
  {
    return Angle.radians(axeMath.mathToMapAngle(this.radians));
  }

  /** Converts this 'map' angle to a 'math' angle.
   *
   * A 'math' angle starts at 3 o'clock and increases counterclockwise.
   * A 'map' angle starts at 12 o'clock and increases clockwise.
   *
   * @return A new Angle object which is the equivalent 'math' angle.
   */
  mapToMathAngle() : Angle
  {
    return Angle.radians(axeMath.mapToMathAngle(this.radians));
  }

  /** Creates a new Angle object from a JSON-ready object.
   *
   * @param jsonObj An object created by a JSON parser from an encoded Angle.
   *
   * @return A new Angle object.
   */
  static fromJson(jsonObj: any) : Angle
  {
    return Angle.degrees(jsonObj.degrees);
  }

  /** Creates a new JSON-ready object from this Angle.
   *
   * @return A JSON-ready object.
   */
  toJson() : ClassJsonObject
  {
    let obj = {
      '_class': 'Angle',
      'degrees': this.degrees
    };

    return obj;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames = [ 'degrees', 'radians' ];
    return axeString.formatObject("Angle", this, propertyNames);
  }

} // END class Angle
