//------------------------------------------------------------------
//  TestClasses.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeClasses from "../classes";
import * as axeString from "../../util/string";
import { AnyCtor } from "../types";
import { UnitTest } from "../../unittest/UnitTest";
import { UnitTestRunner } from "../../unittest/UnitTestRunner";
//------------------------------------------------------------------

/** Unit test for the Classes class.
 */
export class TestClasses extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new Classes object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** Tests class names.
   */
  testNames() : void
  {
    class A
    {
    }

    class B extends A
    {
      constructor()
      {
        super();
      }
    }

    let name = axeClasses.className;

    let undefinedObj = (<any>Object)['foofoofoo'];

    this.assertEqual("undefined", "undefined", name(undefinedObj));
    this.assertEqual("null", "null", name(null));
    this.assertEqual("3.14", "number", name(3.14));
    this.assertEqual("zero", "number", name(0));
    this.assertEqual("boolean(true)", "boolean", name(true));
    this.assertEqual("A", "A", name(new A()));
    this.assertEqual("B", "B", name(new B()));
    this.assertEqual("[1, 2, 3]", "Array", name([1, 2, 3]));
    this.assertEqual("{}", "Object", name({}));
  }

  /** Tests checking base classes.
   */
  testIdBaseClass() : void
  {
    class A
    {
    }

    class B extends A
    {
      constructor()
      {
        super();
      }
    }

    class C extends B
    {
      c: number;

      constructor(c: number)
      {
        super();
        this.c = c;
      }
    }

    class D extends C
    {
      d: string;

      constructor(c: number, d: string)
      {
        super(c);
        this.d = d;
      }
    }

    this.assertEqual('A, B', true, axeClasses.isBaseClass(A, B));
    this.assertEqual('A, D', true, axeClasses.isBaseClass(A, D));
    this.assertEqual('B, A', false, axeClasses.isBaseClass(B, A));
    this.assertEqual('String, A', false, axeClasses.isBaseClass(String, A));
  }

  /** Tests finding base classes.
   */
  testBaseClasses() : void
  {
    class A
    {
      constructor()
      {
      }
    }

    class B extends A
    {
      constructor()
      {
        super();
      }
    }

    class C extends B
    {
      c: number;

      constructor(c: number)
      {
        super();
        this.c = c;
      }
    }

    class D extends C
    {
      d: string;

      constructor(c: number, d: string)
      {
        super(c);
        this.d = d;
      }
    }

    let expected: Array<AnyCtor> = [];
    let actual = axeClasses.baseClasses(A);
    this.assertEqual('A', expected, actual);

    expected = [A];
    actual = axeClasses.baseClasses(B);
    this.assertEqual('D', expected, actual);

    expected = [C, B, A];
    actual = axeClasses.baseClasses(D);
    this.assertEqual('D', expected, actual);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestClasses

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestClasses());
