//------------------------------------------------------------------
//  ClassJsonDecoder.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
import { ObjectMap } from "../util/types";
import { HttpHeaders } from '@angular/common/http';
import { ClassJsonRegistry } from './ClassJsonRegistry';
import { JsonObj } from './jsonTypes/JsonObj';
import { JsonTypeError } from './jsonTypes/JsonTypeError';
//------------------------------------------------------------------

/** Decodes JSON into an object.
 */
export class ClassJsonDecoder
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonDecoder object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Decodes the specified JSON string into an object.
   *
   * @param jsonStr The string to be decoded.
   *
   * @return A @ClassJsonClass-decorated object.
   */
  decode(jsonStr: string) : any
  {
    let objDict = JSON.parse(jsonStr);

    return this.decodeObject(objDict);
  }

  /** Decodes either (1) a ClassJsonClass object, or (2) a list of such objects.
   */
  decodeObject(objDictOrList: Array<ObjectMap<any>> | ObjectMap<any>) : any
  {
    if ((!objDictOrList) || (typeof objDictOrList != 'object'))
    {
      // This is plain JSON, not an encoded ClassJson object.
      // Just return as-is.
      return objDictOrList;
    }

    let isList: boolean = Array.isArray(objDictOrList);

    // The raw objects.
    // When the response is a scalar, this will contain only one object.
    let objDicts: Array<ObjectMap<any>> =
      isList ? (objDictOrList as Array<ObjectMap<any>>) : [objDictOrList];

    // The decoded objects.
    // When the response is a scalar, this will contain only one object.
    let objs: Array<any> = [];

    for (let objDict of objDicts)
    {
      if (!('_class' in objDict))
      {
        // This is plain JSON, not an encoded ClassJson object.
        // Just return the response as-is.
        return objDictOrList;
      }

      let classJsonName = objDict['_class'];

      if (!ClassJsonRegistry.registry.isRegistered(classJsonName))
      {
        let msg = ('Root object _class "' + classJsonName + '" has not been ' +
                   'registered.  Unable to decode JSON: ' +
                   axeString.format(objDict));
        throw new Error(msg);
      }

      let jsonType = new JsonObj(classJsonName);

      let errorMsg: JsonTypeError = jsonType.validateJson(objDict);
      if (errorMsg)
      {
        let msg = errorMsg.format();
        msg += "\n\nUnable to decode JSON:\n" + axeString.format(objDict);
        throw new Error(msg);
      }

      let obj = jsonType.decode(objDict);

      obj = jsonType.decodeLinks([], obj);

      objs.push(obj);
    }

    if (isList)
    {
      return objs;
    }
    else
    {
      return objs[0];
    }
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("ClassJsonDecoder", this, propertyNames);
  }

} // END class ClassJsonDecoder
