//------------------------------------------------------------------
//  JsonDateTime.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
/// <reference path="../../util/date" />
import * as axeClasses from "../../util/classes";
import * as axeDate from "../../util/date";
import * as axeString from "../../util/string";
/// <reference path="JsonType" />
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
//------------------------------------------------------------------

/** Marks a class attribute as a datetime.
 *
 * This is a Date object in Typescript and an ISO 8601 string in JSON.
 */
export class JsonDateTime extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The format that dates shall have.
   */
  static DATETIME_REG_EXP: RegExp =
    new RegExp('\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}[.]\\d{3}Z');

  /** The format that dates shall have.
   */
  static DATETIME_PATTERN: string = 'YYYY-mm-ddTHH:MM:SS.sssZ';

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonDateTime object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

 /** Checks that the specified value can be converted to JSON.
  *
  * @param value The value to validate.
  *
  * @return None if the value is OK, or a JsonTypeError if there is a problem.
  */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }
    else if (!(value instanceof Date))
    {
      let msg = ('is of type date, ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }
    else if (typeof value != 'string')
    {
      let msg = ('is of type datetime (string), ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else if (!this.canBeParsed(value))
    {
      let msg = ('is of type datetime, and the value is not valid format "' +
                 JsonDateTime.DATETIME_PATTERN + '.sssZ"' +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Returns True if the specified string can be parsed as a date.
   */
  private canBeParsed(dateStr: string) : boolean
  {
    return JsonDateTime.DATETIME_REG_EXP.test(dateStr);
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let dateTimeValue: Date = <Date>value;
    return axeDate.formatDateTime(dateTimeValue);
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let stringValue: string = <string>value;
    return axeDate.parseDateTime(stringValue);
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    return value;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("JsonDateTime", this, propertyNames);
  }

} // END class JsonDateTime
