//------------------------------------------------------------------
//  JsonDuration.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
/// <reference path="../../date/Duration" />
import { Duration } from '../../date/Duration';
import * as axeClasses from "../../util/classes";
import * as axeString from "../../util/string";
/// <reference path="JsonType" />
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
//------------------------------------------------------------------

/** Marks a class attribute as a duration.
 *
 * This is a Duration in Typescript and an integer number of ms in JSON.
 */
export class JsonDuration extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonDuration object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }
    if (!(value instanceof Duration))
    {
      let msg = ('is of type duration, ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }

    let duration = <Duration>value;
    let millis: number = duration.milliseconds;

    // Enforce that a number of milliseconds is an integer.
    if (!(isFinite(millis) && Math.floor(millis) == millis))
    {
      let msg = ('is of type duration. The value has a fractional number ' +
                 'of milliseconds.  Value: ' + millis);
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (!value)
    {
      return null;
    }
    else if (typeof value != 'number')
    {
      let msg = ('is of type duration (number), ' +
                 'but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }

    // Enforce that a number of milliseconds is an integer.
    else if (!(isFinite(value) && Math.floor(value) == value))
    {
      let msg = ('is of type duration. The value has a fractional number ' +
                 'of milliseconds.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }
    else
    {
      let valueObj = <Duration>value;
      return valueObj.milliseconds;
    }
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }
    else
    {
      let valueNumber = <number>value;
      return new Duration(valueNumber);
    }
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    return value;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("JsonDuration", this, propertyNames);
  }

} // END class JsonDuration
