//------------------------------------------------------------------
//  JsonTypeError.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../util/string";
//------------------------------------------------------------------

/**An error that prevents an object/JSON encoding or decoding.
 */
export class JsonTypeError
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  // The path into the JSON document that is the error location.
  path: string = '';

  // The error message.
  message: string;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonTypeError object.
   */
  constructor(message: string)
  {
    this.message = message;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Adds a string element to the path.
   *
   *  @param element A string to be added to the path.
   */
  prependPath(element: string) : void
  {
    // Before a list/dictionary subscript, no slash is needed.
    // Otherwise, prepend a slash.

    if (this.path && this.path[0] != '[')
    {
      element += '/';
    }

    this.path = element + this.path;
  }

  /** Adds a numeric index to the path.
   *
   * @param element A string to be added to the path.
   */
  prependPathIndex(index: number) : void
  {
    let indexStr: string = '[' + index + ']';

    if (this.path && this.path[0] != '[')
    {
      indexStr += '/';
    }

    this.path = indexStr + this.path;
  }

  /** Adds a dictionary key to the path.
   *
   * @param element A string to be added to the path.
   */
  prependPathKey(key: string) : void
  {
    let keyStr: string = '[' + key + ']';

    if (this.path && this.path[0] != '[')
    {
      keyStr += '/';
    }

    this.path = keyStr + this.path;
  }

  /** Formats this object into a pleasant message.
   */
  format() : string
  {
    return 'The element at location /' + this.path + ' ' + this.message;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      "path",
      "message"
    ];
    return axeString.formatObject("JsonTypeError", this, propertyNames);
  }

} // END class JsonTypeError
