//------------------------------------------------------------------
//  UnitTest.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
import * as axeFunc from '../util/func';
import * as axeArray from '../util/array';
import * as axeClasses from '../util/classes';
import { Map } from '../util/Map';
import { Pair } from '../util/Pair';
import { Day } from '../date/Day';
import { Duration } from '../date/Duration';
import { ObjectMap } from '../util/types';
import { AnyFunction } from '../util/types';
//------------------------------------------------------------------

type TestMethod = () => void;
interface TestClass
{
  [name: string]: TestMethod;
}

/** An xUnit unit test class.
 */
export class UnitTest
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new UnitTest object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Throws a test fail exception if the values are not equals.
   */
  assertEqual<T>(msg: string, expected: T, actual: T) : void
  {
    let errorMsg = this.checkEqual(expected, actual);
    if (errorMsg != null)
    {
      errorMsg = msg + "\nassertEqual failed.\n" + errorMsg;
      console.error(errorMsg);
      throw new Error(errorMsg);
    }
  }

  /** Fails the test if the two floating point values are not almost equal.
   */
  assertClose(msg: string, expected: number, actual: number) : void
  {
    let diff = expected - actual;
    if (diff < 0)
    {
      diff *= -1;
    }

    if (isNaN(diff) || diff > 1e-12)
    {
      let errorMsg = msg;
      errorMsg += "\nExpected: " + axeString.format(expected);
      errorMsg += "\nActual:   " + axeString.format(actual);

      console.error(errorMsg);
      throw new Error(errorMsg);
    }
  }

  /** Throws a test fail exception if the value is not null.
   */
  assertNull(msg: string, actual: any) : void
  {
    if (actual !== null)
    {
      let errorMsg = msg + "\nassertNull failed.\n";
      errorMsg += "Actual:   " + axeString.format(actual);

      console.error(errorMsg);
      throw new Error(errorMsg);
    }
  }

  /** Throws a test fail exception if the value is not null.
   */
  assertNotNull(msg: string, actual: any) : void
  {
    if (actual === null)
    {
      let errorMsg = msg + "\nassertNotNull failed.";

      console.error(errorMsg);
      throw new Error(errorMsg);
    }
  }

  /** Throws a test fail exception if the object is not of the specified type.
   */
  assertInstanceOf(msg: string, expectedType: any, actualObj: any) : void
  {
    if (!(actualObj instanceof expectedType))
    {
      msg += ('\nassertInstanceOf failed.\n' +
              'Expected type: ' + expectedType.constructor.name + '\n' +
              'Actual type: ' + axeClasses.className(actualObj) + '\n' +
              'Actual object: ' + axeString.format(actualObj));

      console.error(msg);
      throw new Error(msg);
    }
  }

  /** Throws a test fail exception if no exception is thrown.
   */
  assertException(msg: string, testFunc: AnyFunction) : void
  {
    let wasException = false;

    try
    {
      testFunc();
    }
    catch (ex)
    {
      wasException = true;
    }

    if (!wasException)
    {
      console.error(msg);
      throw new Error(msg);
    }
  }

  /** Throws a test fail exception if the value is non-true.
   */
  assert(msg: string, value: any) : void
  {
    if (!value)
    {
      let errorMsg = msg + "\nTest failed:  non-true value.  Value: ";
      errorMsg += axeString.format(value);

      console.error(errorMsg);
      throw new Error(errorMsg);
    }
  }

  /** Throws a test fail exception.
   */
  fail(msg: string) : void
  {
    let errorMsg = msg;

    console.error(errorMsg);
    throw new Error(errorMsg);
  }

  /** Returns an error message if both values are different types.
   */
  private checkAreSameType(first: any, second: any) : string
  {
    if (first === null && second !== null)
    {
      return "Expected null, but got type '" + typeof second + "'.";
    }

    if (typeof first != typeof second)
    {
      return "Expected type '" + axeClasses.className(first) + "', " +
        "but got type '" + axeClasses.className(second) + "'.";
    }

    if (typeof first == "object")
    {
      let firstName = axeClasses.className(first);
      let secondName = axeClasses.className(second);

      if (firstName != secondName)
      {
        return "Expected object of class '" + firstName +
          "', but got class '" + secondName + "'.";
      }
    }

    return null;
  }

  /** Returns an error message if the two values are not equal.
   */
  checkEqual(expected: any, actual: any) : string
  {
    let errorMsg = this.checkAreSameType(expected, actual);
    if (errorMsg)
    {
      let msg = errorMsg;
      msg += "\n";
      msg += "Expected: " + axeString.format(expected) + "\n";
      msg += "Actual:   " + axeString.format(actual);
      return msg;
    }

    // Primitives.

    if (typeof expected == 'string')
    {
      return this.checkStringsEqual(<string><any>expected, <string><any>actual);
    }
    else if (typeof expected == "number" ||
             typeof expected == "boolean")
    {
      if (expected == actual)
      {
        return null;
      }
      else
      {
        let msg = "";
        msg += "Expected: " + axeString.format(expected) + "\n";
        msg += "Actual:   " + axeString.format(actual);
        return msg;
      }
    }

    // Semi-primitive classes that have a meaningful 'valueOf()' method.

    let valueOfClasses: Array<new (...args: any[]) => any> =
      [Date, Day, Duration];

    for (let valueOfClass of valueOfClasses)
    {
      if (expected instanceof valueOfClass)
      {
        if (expected.valueOf() == actual.valueOf())
        {
          return null;
        }
        else
        {
          let msg = "";
          msg += "Expected: " + expected.toString() + "\n";
          msg += "Actual:   " + actual.toString();
          return msg;
        }
      }
    }

    // Arrays.

    if (expected instanceof Array)
    {
      return this.checkArraysEqual(expected, actual);
    }
    else if (expected instanceof Map)
    {
      return this.checkMapsEqual(expected, actual);
    }

    // Objects.

    return this.checkObjectsEqual(expected, actual);
  }

  /** Returns an error message if the arrays are not equal.
   */
  checkArraysEqual(expected: Array<any>, actual: Array<any>) : string
  {
    if (expected.length != actual.length)
    {
      let msg = "Expected array length " + expected.length + ", actually got " +
        "array of length " + actual.length + ".";
      msg += "\nexpected array: " + axeString.format(expected);
      msg += "\nactual array:   " + axeString.format(actual);
      return msg;
    }

    for (let i = 0; i < expected.length; ++i)
    {
      let errorMsg = this.checkAreSameType(expected[i], actual[i]);
      if (errorMsg)
      {
        errorMsg = "Incorrect value for item " + i + " of array.\n" + errorMsg;
        errorMsg += "\nexpected array: " + axeString.format(expected);
        errorMsg += "\nactual array:   " + axeString.format(actual);
        return errorMsg;
      }
    }

    return null;
  }

  /** Returns an error message if the maps are not equal.
   */
  checkMapsEqual(expected: Map<any, any>, actual: Map<any, any>) : string
  {
    for (let key of expected.keys())
    {
      if (!actual.hasKey(key))
      {
        let msg = "Expected key '" + key + "' missing from map.\n";
        msg += "Expected map: " + expected.toString() + "\n";
        msg += "Actual map:   " + actual.toString();
        return msg;
      }
    }
    for (let key of actual.keys())
    {
      if (!expected.hasKey(key))
      {
        let msg = "Extra key '" + key + "' in map.\n";
        msg += "Expected map: " + expected.toString() + "\n";
        msg += "Actual map:   " + actual.toString();
        return msg;
      }
    }

    for (let key of expected.keys())
    {
      let errorMsg = this.checkEqual(expected.get(key), actual.get(key));
      if (errorMsg)
      {
        let msg = "Wrong value for key '" + key + "'.\n";
        msg += errorMsg + "\n";
        msg += "Expected map: " + expected.toString() + "\n";
        msg += "Actual map:   " + actual.toString();
        return msg;
      }
    }

    return null;
  }

  /** Returns an error message if the objects are not equal.
   */
  checkObjectsEqual(expected: ObjectMap<any>, actual: ObjectMap<any>) : string
  {
    // Make lists of the property names for each object.
    let expectedKeys: Array<string> = [];
    let actualKeys: Array<string> = [];

    for (let key in expected)
    {
      expectedKeys.push(key);
    }
    for (let key in actual)
    {
      actualKeys.push(key);
    }

    for (let key of expectedKeys)
    {
      if (!axeArray.contains(actualKeys, key))
      {
        let msg = "Expected key '" + key + "' missing from object.\n";
        msg += "Expected object: " + axeString.format(expected) + "\n";
        msg += "Actual object:   " + axeString.format(actual) + "\n";
        return msg;
      }
    }
    for (let key of actualKeys)
    {
      if (!axeArray.contains(expectedKeys, key))
      {
        let msg = "Extra key '" + key + "' in map.\n";
        msg += "Expected object: " + axeString.format(expected) + "\n";
        msg += "Actual object:   " + axeString.format(actual) + "\n";
        return msg;
      }
    }

    for (let key of expectedKeys)
    {
      let errorMsg = this.checkEqual(expected[key], actual[key]);
      if (errorMsg)
      {
        errorMsg = "Wrong value for property: " + key + "\n" + errorMsg;
        errorMsg += "\n";
        errorMsg += "Expected object: " + axeString.format(expected) + "\n";
        errorMsg += "Actual object:   " + axeString.format(actual) + "\n";

        return errorMsg;
      }
    }

    return null;
  }

  /** Returns an error message if the strings are not equals.
   */
  checkStringsEqual(expected: string, actual: string) : string
  {
    let minLength = Math.min(expected.length, actual.length);

    for (let index = 0; index < minLength; ++index)
    {
      if (expected.charAt(index) != actual.charAt(index))
      {
        let width = 25;
        let begin = Math.max(0, index - width);
        let endExpected = Math.min(expected.length, index + width);
        let endActual = Math.min(actual.length, index + width);

        let beginTruncated = begin > 0;
        let endExpectedTruncated = endExpected < expected.length;
        let endActualTruncated = endActual < actual.length;

        // The index relative to the 'begin' position.
        let relativeIndex = index - begin;

        let expectedSnippet = expected.substring(begin, endExpected);
        let actualSnippet = actual.substring(begin, endActual);
        let differsSnippet = axeString.repeat(" ", relativeIndex) + "^";
        if (beginTruncated)
        {
          expectedSnippet = "..." + expectedSnippet;
          actualSnippet = "..." + actualSnippet;
          differsSnippet = "   " + differsSnippet;
        }
        if (endExpectedTruncated)
        {
          expectedSnippet += "...";
        }
        if (endActualTruncated)
        {
          actualSnippet += "...";
        }

        let errorMsg = "";
        errorMsg += "Strings differ at position " + index + "\n";
        errorMsg += "Expected: " + expectedSnippet + "\n";
        errorMsg += "Actual:   " + actualSnippet  + "\n";
        errorMsg += "Differs:  " + differsSnippet + "\n";
        errorMsg += "\n";
        errorMsg += "Full expected string: " + expected + "\n";
        errorMsg += "Full actual string:   " + actual + "\n";
        errorMsg += "Differs:              " +
          axeString.repeat(" ", index) + "^\n";

        return errorMsg;
      }
    }

    if (expected.length > actual.length)
    {
      let errorMsg = "";
      errorMsg += "Actual string too short.  Missing text: ...";
      errorMsg += expected.substring(actual.length, expected.length);
      return errorMsg;
    }
    if (expected.length < actual.length)
    {
      let errorMsg = "";
      errorMsg += "Actual string too long.  Extra text: ...";
      errorMsg += actual.substring(expected.length, actual.length);
      return errorMsg;
    }

    return null;
  }

  /** Runs all test methods named 'testXxx()'.
   *
   * @return An error message listing all failed tests.
   *         Will be null if all tests passed.
   */
  run() : string
  {
    let className: string = axeClasses.className(this);

    let allPassed: boolean = true;
    let statusMsg: string = "";

    for (let testFunc of this.findTests())
    {
      let funcName = testFunc.first;
      let func = axeFunc.bind2(this, testFunc.second);

      statusMsg += "Executing " + className + "." + funcName + "...";

      try
      {
        func();

        statusMsg += "PASSED\n";
      }
      catch (ex)
      {
        allPassed = false;
        statusMsg += "FAILED\n";
        statusMsg += ex.stack + "\n";
      }
    }

    if (allPassed)
    {
      console.log(statusMsg);
    }
    else
    {
      console.error(statusMsg);
    }

    return allPassed ? null : statusMsg;
  }

  /** Finds all test methods name 'testXxx()'.
   */
  findTests() : Array<Pair<string, TestMethod>>
  {
    let testClass = <TestClass><any>this;

    let funcs: Array<Pair<string, TestMethod>> = [];

    for (let functionName of axeClasses.methodNames(this))
    {
      if (axeString.startsWith(functionName, 'test'))
      {
        let func: TestMethod = testClass[functionName];
        funcs.push(new Pair(functionName, func));
      }
    }

    return funcs;
  }

} // END class UnitTest
