//------------------------------------------------------------------
//  string.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

import * as axeClasses from './classes';
import { Map } from './Map';

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------


/** Returns true if the string ends with the specified suffix.
 */
export function endsWith(s: string, suffix: string) : boolean
{
  if (s.length < suffix.length) {
    return false;
  }

  return s.substring(s.length - suffix.length) == suffix;
}

/** Returns true if the string starts with the specified prefix.
 */
export function startsWith(s: string, prefix: string) : boolean
{
  if (s.length < prefix.length) {
    return false;
  }

  return s.substring(0, prefix.length) == prefix;
}

/** Returns a substring of the input string.
 *
 * Like string.substring(), but also accepts negative endIndex, a la Python.
 *
 * A beginIndex less than zero is treated as zero.
 * An endIndex greater than length() is treated as length().
 *
 * @param beginIndex The first character to return.
 * @param endIndex One past the last character to return.
 *
 * @return The substring.
 */
export function substring(s: string,
                          beginIndex: number,
                          endIndex: number) : string
{
  // Translate a negative index to positive.
  if (beginIndex < 0)
  {
    beginIndex += s.length;
  }
  if (endIndex < 0)
  {
    endIndex += s.length;
  }

  // Bring indexes within [0..length()].

  beginIndex = Math.max(0, beginIndex);
  beginIndex = Math.min(s.length, beginIndex);

  endIndex = Math.max(0, endIndex);
  endIndex = Math.min(s.length, endIndex);

  // Handle the zero-length case that string.substring() will choke on.

  if (beginIndex >= endIndex)
  {
    return "";
  }

  return s.substring(beginIndex, endIndex);
}

/** Returns the input string with leading and trailing whitespace removed.
 */
export function strip(s: string) : string
{
  let first = 0;
  while (first < s.length) {
    if (isWhite(s.charAt(first))) {
      first++;
    } else {
      break;
    }
  }

  let last = s.length;
  while (last > first) {
    if (isWhite(s.charAt(last - 1))) {
      last--;
    } else {
      break;
    }
  }

  return s.substring(first, last);
}

/** Returns true if the string is a single whitespace character.
 */
export function isWhite(s: string) : boolean
{
  return (s == " " || s == "\n" || s == "\r" || s == "\t");
}

/** Returns true if the string is one or more digits.
 */
export function isDigits(s: string) : boolean
{
  if (s.length == 0) {
    return false;
  }

  for (let i = 0; i < s.length; ++i) {
    if (!isDigit(s.charAt(i))) {
      return false;
    }
  }

  return true;
}

/** Returns true if the string is a single digit.
 */
export function isDigit(s: string) : boolean
{
  return (s == "0" ||
          s == "1" ||
          s == "2" ||
          s == "3" ||
          s == "4" ||
          s == "5" ||
          s == "6" ||
          s == "7" ||
          s == "8" ||
          s == "9");
}

/** Returns a string repeated a specified number of times.
 *
 * @param s The string to repeat.
 * @param n The number of time to repeat it.
 *
 * @return A string that is the repeated strings.
 */
export function repeat(s: string, n: number) : string
{
  let ret = "";

  for (let i = 0; i < n; ++i) {
    ret += s;
  }

  return ret;
}

/** Replace all occurances of a substring with a new string.
 *
 * @param text The text to replace in.
 * @param oldSubstring The text to replace.
 * @param newSubstring The text to replace with.
 */
export function replace(text: string,
                        oldSubstring: string,
                        newSubstring: string) : string
{
  let regExpText = escapeRegExp(oldSubstring);
  let regExp = RegExp(regExpText, 'g');

  return text.replace(regExp, newSubstring);
}

/** Escapes a regular expression string.
 *
 * Passing a string through this function makes is safe to pass into the
 * RegExp constructor.
 */
export function escapeRegExp(regExpStr: string) : string
{
  return regExpStr.replace(/([.*+?^=!:${}()|\[\]\/\\])/g, "\\$1");
}

/** Capitalizes the first letter of the string.
 */
export function capitalize(s: string) : string
{
  if (s.length == 0) {
    return s;
  } else if (s.length == 1) {
    return s.toUpperCase();
  } else {
    return (s.substring(0, 1).toUpperCase() + s.substring(1));
  }
}

/** Makes the first letter of the string lower case.
 */
export function decapitalize(s: string) : string
{
  if (s.length == 0) {
    return s;
  } else if (s.length == 1) {
    return s.toLowerCase();
  } else {
    return (s.substring(0, 1).toLowerCase() + s.substring(1));
  }
}

/** Splits a string into an array.
 *
 * Different from String.split() because it will return an empty
 * array when the string is empty.  String.split() returns an array
 * with one empty string when the string is empty.
 *
 * @param stringList The string to be split.
 * @param separator The separator character.
 */
export function splitNonEmpty(stringList: string,
                              separator: string) : Array<string>
{
  if (stringList == "") {
    return [];
  } else {
    return stringList.split(separator);
  }
}

/** Returns the first line of the string.
 *
 * @param lines The lines to trim.
 * @param maxLen The string will be trimmed to this length.
 */
export function firstLine(lines: string, maxLen: number) : string
{
  // Just return undefined and null.
  if (!lines) {
    return lines;
  }

  let newlineIdx = lines.indexOf('\n');

  if (newlineIdx == -1) {
    return lines;
  }

  if (maxLen >= 3 && maxLen < newlineIdx) {
    newlineIdx = maxLen - 3;
  }

  return lines.substring(0, newlineIdx) + '...';
}

/** Tests whether a string is all upper case.
 */
export function isUpperCase(input: string) : boolean
{
  return (input == input.toUpperCase());
}

/** Tests whether a string is all lower case.
 */
export function isLowerCase(input: string) : boolean
{
  return (input == input.toLowerCase());
}

/** Converts a camel-case identifier to a dash-separated set of lowercase words.
 *
 * For example, fastCar becomes fast-car.
 */
export function camelCaseToDashes(text: string) : string
{
  text = camelCaseToWords(text);
  text = text.toLowerCase();
  text = replace(text, ' ', '-');

  return text;
}

/** Converts a camel-case identifier to a space-separated set of words with the
 *  initial character capitalized; e.g.,
 *  camelCaseInputString -> Camel Case Input String
 */
export function camelCaseToWords(text: string) : string
{
  for (let i = 1; i < text.length - 1; i++) {
    if (isLowerCase(text.charAt(i - 1)) &&
        isUpperCase(text.charAt(i)) &&
        isLowerCase(text.charAt(i + 1))) {
      text = text.slice(0, i) + " " + text.slice(i);
      i++;
    }
  }

  return capitalize(text);
}

/** Converts a space-separated set of words to a camel-case identifier; e.g.,
 *  Space Separated Input String -> spaceSeparatedInputString
 */
export function wordsToCamelCase(text: string) : string
{
  let words = text.split(" ");

  let output = "";

  for (let i = 0; i < words.length; i++) {
    let word = strip(words[i]);

    output = output + capitalize(word);
  }

  return decapitalize(output);
}

/** Concatenate a number with the appropriate ordinal suffix.
 *
 * @param num An int.
 *
 * @return A string.
 */
export function formatOrdinal(num: number) : string
{
  let tens = num % 10;
  let hundreds = num % 100;

  if (tens == 1 && hundreds != 11)
  {
    return num + "st";
  }
  if (tens == 2 && hundreds != 12)
  {
    return num + "nd";
  }
  if (tens == 3 && hundreds != 13)
  {
    return num + "rd";
  }
  return num + "th";
}

/** Formats an object into a string repreresentation.
 */
export function formatObject(className: string,
                             obj: any,
                             propertyNames: Array<string>,
                             propertyValues?: Map<string, string>) : string
{
  let propStrs: Array<string> = [];

  for (let i = 0; i < propertyNames.length; ++i)
  {
    let propertyName = propertyNames[i];

    let propValue: string = "";
    if (propertyValues && propertyValues.hasKey(propertyName))
    {
      propValue = propertyValues.get(propertyName);
    }
    else
    {
      propValue = format(obj[propertyName]);
    }

    propStrs.push(propertyName + "=" + propValue);
  }

  return className + "(" + propStrs.join(", ") + ")";
}

/** Formats an object more nicely than toString().
 *
 * Puts quotations around strings.
 */
export function format(object: any) : string
{
  if (typeof object == 'undefined')
  {
    return "undefined";
  }
  else if (object === null)
  {
    return "null";
  }
  else if (typeof object == 'number' ||
           typeof object == 'boolean')
  {
    return object.toString();
  }
  else if (typeof object == 'string')
  {
    return '"' + object + '"';
  }
  else if (object instanceof Array)
  {
    let theArray = <Array<any>>object;

    let items: Array<string> = [];
    for (let item of theArray)
    {
      items.push(format(item));
    }
    return "[" + items.join(", ") + "]";
  }
  else if (object.toString() != "[object Object]")
  {
    // The object's toString() method has been overriden, so use that.
    return object.toString();
  }
  else
  {
    // Instead of returning unhelpful '[object Object]',
    // return a list of properties.

    let className: string = axeClasses.className(object);

    let names: Array<string> = [];
    for (let propName in object)
    {
      names.push(propName);
    }
    return formatObject(className, object, names);
  }
}

