//------------------------------------------------------------------
//  JsonList.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from "../../util/array";
import * as axeClasses from "../../util/classes";
import * as axeString from "../../util/string";
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
//------------------------------------------------------------------

/** Marks a class attribute as a list of a ClassJson type.
 */
export class JsonList extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  // The type of the objects in this list.
  itemType: JsonType;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonList object.
   */
  constructor(itemType: JsonType)
  {
    super();
    this.itemType = itemType;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    if (!(value instanceof Array))
    {
      let msg = ('is of type list but the value is of type ' +
                 axeClasses.className(value) + '.  Value: ' +
                 axeString.format(value));
      return new JsonTypeError(msg);
    }

    for (let i = 0; i < value.length; ++i)
    {
      let item = value[i];
      let errorMsg = this.itemType.validate(item);

      if (errorMsg)
      {
        errorMsg.prependPathIndex(i);
        return errorMsg;
      }
    }

    return null;
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    if (!(value instanceof Array))
    {
      let msg = ('is of type list but the value is of type ' +
                 axeClasses.className(value) + '.  Value: ' +
                 axeString.format(value));
      return new JsonTypeError(msg);
    }

    for (let i = 0; i < value.length; ++i)
    {
      let item = value[i];
      let errorMsg = this.itemType.validateJson(item);

      if (errorMsg)
      {
        errorMsg.prependPathIndex(i);
        return errorMsg;
      }
    }

    return null;
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let valueList: Array<any> = <Array<any>>value;

    let ret: Array<any> = [];

    for (let theValue of valueList)
    {
      ret.push(this.itemType.encode(theValue));
    }

    return ret;
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let valueList: Array<any> = <Array<any>>value;

    let ret: Array<any> = [];

    for (let theValue of valueList)
    {
      ret.push(this.itemType.decode(theValue));
    }

    return ret;
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    if (value === null)
    {
      return null;
    }

    else if (!axeArray.isArray(value))
    {
      throw new Error("Expected list, got: " + axeClasses.className(value));
    }

    let valueList: Array<any> = <Array<any>>value;

    for (let i = 0; i < valueList.length; ++i)
    {
      let valueItem = valueList[i];
      valueList[i] = this.itemType.decodeLinks(parents, valueItem);
    }

    return valueList;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      'itemType'
    ];
    return axeString.formatObject("JsonList", this, propertyNames);
  }

} // END class JsonList
