//------------------------------------------------------------------
//  JsonMap.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeClassJson  from '../classJson';
import { BooleanMapType } from '../../util/Map';
import { DateMapType } from '../../util/Map';
import { DurationMapType } from '../../util/Map';
import { DayMapType } from '../../util/Map';
import { IntMapType } from '../../util/Map';
import { Map } from '../../util/Map';
import { MapKeyType } from '../../util/Map';
import { StringMapType } from '../../util/Map';
import * as axeClasses from "../../util/classes";
import * as axeString from "../../util/string";
import { JsonDate } from './JsonDate';
import { JsonDateTime } from './JsonDateTime';
import { JsonDuration } from './JsonDuration';
import { JsonObj } from './JsonObj';
import { JsonPrimitiveType } from './JsonPrimitiveType';
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
//------------------------------------------------------------------

/** Add class documentation here.
 */
export class JsonMap extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The type of keys in this list.
   */
  keyType: JsonType;

  /** The type of values in this list.
   */
  valueType: JsonType;

  /** The object used a key encoder for lists produced by this object.
   */
  mapKeyType: MapKeyType<any>;

  /** Map types to use for objects.
   */
  static classMapKeyTypes: Map<string, MapKeyType<any>> =
    Map.createStringMap<MapKeyType<any>>();

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonMap object.
   */
  constructor(keyType: JsonType, valueType: JsonType)
  {
    super();

    this.keyType = keyType;
    this.valueType = valueType;

    this.mapKeyType = this.chooseMapKeyType(this.keyType);
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Sets the MapKeyType to be used for the specified class.
   */
  static setMapKeyType(className: string, mapKeyType: MapKeyType<any>) : void
  {
    JsonMap.classMapKeyTypes.put(className, mapKeyType);
  }

  /** Chooses the correct MapKeyType for the specified JsonType.
   */
  chooseMapKeyType(jsonType: JsonType) : MapKeyType<any>
  {
    if (jsonType instanceof JsonDate)
    {
      return new DayMapType();
    }
    else if (jsonType instanceof JsonDateTime)
    {
      return new DateMapType();
    }
    else if (jsonType instanceof JsonDuration)
    {
      return new DurationMapType();
    }
    else if (jsonType instanceof JsonPrimitiveType)
    {
      let primitiveType = <JsonPrimitiveType>jsonType;
      let typeName: string = primitiveType.typeName;

      if (typeName == 'string')
      {
        return new StringMapType();
      }
      if (typeName == 'int')
      {
        return new IntMapType();
      }
      if (typeName == 'bool')
      {
        return new BooleanMapType();
      }
      else
      {
        throw new Error("Unsupported primitive type: " + typeName);
      }
    }
    else if (jsonType instanceof JsonObj)
    {
      let jsonObj = <JsonObj>jsonType;
      let className: string = jsonObj.className;

      if (!JsonMap.classMapKeyTypes.hasKey(className))
      {
        let msg = "No mapKeyType has been registered for class: " + className;
        throw new Error(msg);
      }
      else
      {
        return JsonMap.classMapKeyTypes.get(className);
      }
    }
    else
    {
      throw new Error("Unsupported type: " + axeClasses.className(jsonType));
    }
  }

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    if (!(value instanceof Map))
    {
      let msg = ('is type dict but the value is of type ' +
                 axeClasses.className(value) + '.  Value: ' +
                 axeString.format(value));
      return new JsonTypeError(msg);
    }

    let srcMap: Map<any, any> = <Map<any, any>>value;

    for (let key of srcMap.keys())
    {
      let mapValue = srcMap.get(key);

      let keyErrorMsg = this.keyType.validate(key);
      let valueErrorMsg = this.valueType.validate(mapValue);

      if (keyErrorMsg)
      {
        keyErrorMsg.prependPathKey(key);
        return keyErrorMsg;
      }
      if (valueErrorMsg)
      {
        valueErrorMsg.prependPathKey(key);
        return valueErrorMsg;
      }
    }
    return null;
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    let valueList: Array<any> = <Array<any>>value;

    for (let i = 0; i < valueList.length; ++i)
    {
      let item = valueList[i];
      if (item.length != 2)
      {
        let msg = ("list item " + i + " " +
                   "should be length 2, but item is length " + item.length  +
                   ". Raw values: " + axeString.format(value));
        return new JsonTypeError(msg);
      }

      let key = item[0];
      let mapValue = item[1];
      let keyErrorMsg = this.keyType.validateJson(key);
      let valueErrorMsg = this.valueType.validateJson(mapValue);

      if (keyErrorMsg)
      {
        keyErrorMsg.prependPathKey(key);
        return keyErrorMsg;
      }
      if (valueErrorMsg)
      {
        valueErrorMsg.prependPathKey(key);
        return valueErrorMsg;
      }
    }

    return null;
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let valueDict: Map<any, any> = <Map<any, any>>value;

    // Convert the Map to a list.
    let theList: Array<any> = valueDict.toArrays();

    // Encode the keys and values.
    for (let i = 0; i < theList.length; ++i)
    {
      let keyValue = theList[i];

      axeClassJson.assert(keyValue.length == 2, keyValue);

      keyValue[0] = this.keyType.encode(keyValue[0]);
      keyValue[1] = this.valueType.encode(keyValue[1]);
    }

    return theList;
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let valueList: Array<[any, any]> = <Array<[any, any]>>value;

    let decodedList: Array<[any, any]> = valueList.slice();

    // Decode the keys and values.
    for (let i = 0; i < valueList.length; ++i)
    {
      let keyValue = valueList[i];

      axeClassJson.assert(keyValue.length == 2, keyValue);

      let encodedKey = this.keyType.decode(keyValue[0]);
      let encodedValue = this.valueType.decode(keyValue[1]);

      decodedList.push([encodedKey, encodedValue]);
    }

    return Map.fromArrays(decodedList, this.mapKeyType);
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    if (value === null)
    {
      return null;
    }

    else if (!(value instanceof Map))
    {
      throw new Error("Expected Map, got: " + axeClasses.className(value));
    }

    let valueMap = <Map<any, any>>value;

    for (let item of valueMap.items())
    {
      let newValue = this.valueType.decodeLinks(parents, item.value);
      if (newValue !== item.value)
      {
        valueMap.put(item.key, newValue);
      }
    }

    return valueMap;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      "keyType",
      "valueType"
    ];
    return axeString.formatObject("JsonMap", this, propertyNames);
  }

} // END class JsonMap
