//------------------------------------------------------------------
//  Map.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from './array';
import * as axeString from "./string";
import * as axeDate from "./date";
import { ObjectMap } from './types';
import { MapSetterValue } from './MapSetterValue';
import { Day } from '../date/Day';
import { Duration } from '../date/Duration';
//------------------------------------------------------------------

/** A mapping of keys to values.
 *
 * This map remembers the order in which items were added,
 * and returns them in that order.
 */
export class Map<T, U>
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The KeyType object used to encode keys to strings.
   */
  keyType: MapKeyType<T>;

  /** Private.  The keys in this map, in the order they were added.
   */
  keyList: Array<string> = [];

  /** Private.  The index of each key in keyList, for faster lookup.
   */
  keyIndexes: KeyIndexMap = {};

  /** Private.  The underlying map storage object.
   */
  mapObject: StorageMap<U> = {};

  /** Private.  The original un-encoded key values.
   */
  rawKeys: RawKeyMap<T> = {};

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Initializes a new Map object.
   */
  constructor(keyType: MapKeyType<T>)
  {
    this.keyType = keyType;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Creates a map from the specified object's values.
   */
  static asMap<U>(objectMap: ObjectMap<U>) : Map<string, U>
  {
    let ret: Map<string, U> = Map.createStringMap<U>();

    for (let key in objectMap)
    {
      ret.put(key, objectMap[key]);
    }

    return ret;
  }

  /** Creates a new map that has string keys.
   */
  static createStringMap<U>() : Map<string, U>
  {
    return new Map<string, U>(new StringMapType());
  }

  /** Creates a new map that has int keys.
   */
  static createIntMap<U>() : Map<number, U>
  {
    return new Map<number, U>(new IntMapType());
  }

  /** Puts a value in the map.
   *
   * An existing value is replaced.
   */
  put(key: T, value: U) : void
  {
    let keyStr = this.keyType.encode(key);

    if (!this.hasEncodedKey(keyStr))
    {
      this.keyIndexes[keyStr] = this.keyList.length;
      this.keyList.push(keyStr);
    }

    this.rawKeys[keyStr] = key;
    this.mapObject[keyStr] = value;
  }

  /** Copies all key/values from another map to this one.
   */
  putAll(otherMap: Map<T, U>) : void
  {
    let items = otherMap.items();
    for (let i = 0; i < items.length; ++i)
    {
      let item = items[i];
      this.put(item.key, item.value);
    }
  }

  /** Copies all key/values from a list to this one.
   */
  putPairs(pairList: Array<[T, U]>) : void
  {
    for (let i = 0; i < pairList.length; ++i)
    {
      let pair = pairList[i];
      this.put(pair[0], pair[1]);
    }
  }

  /** If key is this map, return it.  Otherwise, insert the default.
   *
   * Instead of doing:
   *
   *   if (myMap.hasKey(foo))
   *   {
   *     myMap.put(foo, []);
   *   }
   *
   *   myMap.get(foo).push('blah');
   *
   * You can do:
   *
   *   myMap.putDefault(foo, []).push('blah');
   *
   * This method is similar to the Python dict's setdefault() method.
   */
  putDefault(key: T, defaultValue: U) : U
  {
    if (this.hasKey(key))
    {
      return this.get(key);
    }
    else
    {
      this.put(key, defaultValue);
      return defaultValue;
    }
  }

  /** Returns a value from the map.
   *
   * Thows an exception if the value is not present.
   */
  get(key: T) : U
  {
    let keyStr: string = this.keyType.encode(key);

    if (!this.hasEncodedKey(keyStr))
    {
      throw new Error("Error: map doesn't contain key: " + key);
    }

    return this.mapObject[keyStr];
  }

  /** Returns a value from the map if present, or a default.
   *
   * @param key The value for this key will be returned.
   * @param defaultValue If the key is not present, this value will be returned.
   */
  getDefault(key: T, defaultValue: U) : U
  {
    let keyStr: string = this.keyType.encode(key);

    if (!this.hasEncodedKey(keyStr))
    {
      return defaultValue;
    }

    return this.mapObject[keyStr];
  }

  /** Removes a key from the map.
   */
  remove(key: T) : void
  {
    let keyStr = this.keyType.encode(key);

    if (this.hasEncodedKey(keyStr))
    {
      delete this.keyIndexes[keyStr];
      axeArray.remove(this.keyList, keyStr);
      delete this.rawKeys[keyStr];
      delete this.mapObject[keyStr];
    }
  }

  /** Removes all keys from the map.
   */
  removeAll() : void
  {
    let keys = this.keys();

    for (let i = 0; i < keys.length; ++i)
    {
      let key: T = keys[i];
      this.remove(key);
    }
  }

  /** Returns a proxy object to get/set the specified key's value as a property.
   *
   * This class is mostly to allow AngularJS ng-model to be used
   * with a Map value.
   *
   * For example, to tell an Angular <input> to get/set the value
   * for the key 3, you would do:
   *
   *   ng-model='myMap.access(3).value'
   *
   * myMap.access(3) returns a MapSetterValue that get/sets the
   * value for the 3 key if you get/set its value property.
   *
   * So this code:
   *
   *   myMap.access(3).value = 42;
   *   let value3 = myMap.access(3).value;
   *
   * is equivalent to:
   *
   *   myMap.put(3, 42);
   *   let value3 = myMap.get(3);
   */
  access(key: T) : MapSetterValue<T, U>
  {
    return new MapSetterValue<T, U>(this, key);
  }

  /** Returns the number of items in this map.
   */
  size() : number
  {
    return this.keyList.length;
  }

  /** Returns true if the key is contained in this map.
   */
  hasKey(key: T) : boolean
  {
    let keyStr: string = this.keyType.encode(key);

    return this.hasEncodedKey(keyStr);
  }

  /** Returns all the keys in this map.
   *
   * @return A list of keys in the order they were last put in the map.
   */
  keys() : Array<T>
  {
    let ret: Array<T> = [];
    for (let i = 0; i < this.keyList.length; ++i)
    {
      ret.push(this.rawKeys[this.keyList[i]]);
    }

    return ret;
  }

  /** Returns all the values in this map.
   *
   * @return A list of values in the order they were last put in the map.
   */
  values() : Array<U>
  {
    let ret: Array<U> = [];
    for (let i = 0; i < this.keyList.length; ++i)
    {
      ret.push(this.mapObject[this.keyList[i]]);
    }

    return ret;
  }

  /** Returns all the values in this map.
   *
   * @return A list of {key: value} in the order they were last put in the map.
   */
  items() : Array<MapKeyValueItem<T, U>>
  {
    let ret: Array<MapKeyValueItem<T, U>> = [];
    for (let i = 0; i < this.keyList.length; ++i)
    {
      let item = { "key" : this.rawKeys[this.keyList[i]],
                   "value" : this.mapObject[this.keyList[i]] };
      ret.push(item);
    }

    return ret;
  }

  /** Returns true if the key is contained in this map.
   */
  hasEncodedKey(keyStr: string) : boolean
  {
    return typeof this.keyIndexes[keyStr] != 'undefined';
  }

  /** Returns an Object with this Map's data in it.
   *
   * An Object can only have string keys, so this will throw an exception
   * if it is called on a Map with non-string keys.
   */
  toObject() : ObjectMap<U>
  {
    if (this.keyType.typeName != 'string')
    {
      let msg = "Called toObject() on a map whose key type is " +
        this.keyType.typeName + ".  This method may only be called on a Map " +
        "whose key type is string.";
      throw new Error(msg);
    }

    let ret: ObjectMap<U> = {};

    for (let item of this.items())
    {
      // This cast is safe because we've already thrown an exception above
      // if the key type is not string.
      let keyStr: string = <string><any>item.key;

      ret[keyStr] = item.value;
    }

    return ret;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let str = '{';

    let items = this.items();
    for (let i = 0; i < this.size(); ++i)
    {
      let item = items[i];

      str += axeString.format(item.key) + ":" + axeString.format(item.value);

      if (i + 1 < this.size())
      {
        str += ",";
      }
    }
    str += "}";

    return str;
  }

  /** Creates a new Map from an array of arrays decoded from JSON.
   */
  static fromArrays<V, W>(arrayArray: Array<[V, W]>,
                          keyType: MapKeyType<V>) : Map<V, W>
  {
    let map: Map<V, W> = new Map<V, W>(keyType);

    for (let i = 0; i < arrayArray.length; ++i)
    {
      let keyValueArray = arrayArray[i];
      if (keyValueArray.length != 2)
      {
        let msg = "Error parsing map.  Item " + i + " has an invalid length ";
        msg += "of " + keyValueArray.length + ". ";
        msg += "Raw data: " + arrayArray;
        throw Error(msg);
      }

      let key: V = <V>keyValueArray[0];
      let value: W = <W>keyValueArray[1];

      map.put(key, value);
    }

    return map;
  }

  /** Returns an array of arrays to use for JSON encoding.
   */
  toArrays() : Array<[T, U]>
  {
    let ret: Array<[T, U]> = [];

    for (let i = 0; i < this.keyList.length; ++i)
    {
      let keyStr = this.keyList[i];
      ret.push([this.rawKeys[keyStr], this.mapObject[keyStr]]);
    }

    return ret;
  }

} // END class Map

//----------------------------------------------------------------
// Helper Classes
//----------------------------------------------------------------

export interface MapKeyValueItem<T, U>
{
  key: T;
  value: U;
}

export interface MapKeyType<T>
{
  typeName: string;
  encode(key: T) : string;
}

export class StringMapType implements MapKeyType<string>
{
  typeName: string;

  constructor()
  {
    this.typeName = "string";
  }

  encode(key: string) : string
  {
    return key;
  }
}

export class IntMapType implements MapKeyType<number>
{
  typeName: string;

  constructor()
  {
    this.typeName = "int";
  }

  encode(key: number) : string
  {
    return "" + key;
  }
}

export class BooleanMapType implements MapKeyType<boolean>
{
  typeName: string;

  constructor()
  {
    this.typeName = "boolean";
  }

  encode(key: boolean) : string
  {
    return "" + key;
  }
}

export class DayMapType implements MapKeyType<Day>
{
  typeName: string;

  constructor()
  {
    this.typeName = "Day";
  }

  encode(key: Day) : string
  {
    return axeDate.formatDate(key);
  }
}

export class DateMapType implements MapKeyType<Date>
{
  typeName: string;

  constructor()
  {
    this.typeName = "Date";
  }

  encode(key: Date) : string
  {
    return axeDate.formatDateTime(key);
  }
}

export class DurationMapType implements MapKeyType<Duration>
{
  typeName: string;

  constructor()
  {
    this.typeName = "Duration";
  }

  encode(key: Duration) : string
  {
    return "" + key.milliseconds;
  }
}

export interface KeyIndexMap
{
  [key: string]: number;
}

export interface RawKeyMap<T>
{
  [key: string]: T;
}

export interface StorageMap<U>
{
  [key: string]: U;
}

