//------------------------------------------------------------------
//  data.js
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

namespace axe.data
{

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Finds the first parent of a node that is of a specified element type.
 *
 * Throws an exception if no such node is found.
 */
export function findAncestor(node: Node,
                             elementName: string) : Node
{
  // Store the body in a local variable to avoid lots of global lookups.
  let theBody: Node = document.body;
  let currNode: Node = node;

  while (true)
  {
    if (currNode == theBody)
    {
      let msg = "Failed to find parent node of type: " + elementName;
      throw Error(msg);
    }
    else if (currNode.nodeName == elementName)
    {
      return currNode;
    }

    currNode = currNode.parentNode;
  }
}

/** Finds the common ancestor siblings of two nodes.
 *
 * Returns the 2 sibling nodes just below the common ancestor node
 * of the specified nodes.  In this case, a length-2 array is returned.
 *
 * If one of the specified nodes is an ancestor of the other,
 * the ancestor node will be returned.  In this case, a length-1 array
 * is returned.
 */
export function ancestorSiblings(node1: Node,
                                 node2: Node) : Array<Node>
{
  let ancestors1 = axe.data.ancestorList(node1);
  let ancestors2 = axe.data.ancestorList(node2);

  let i = 0;

  while (true)
  {
    if (ancestors1[i] !== ancestors2[i])
    {
      return [ ancestors1[i], ancestors2[i] ];
    }

    if (ancestors1.length == i + 1)
    {
      // Node 1 is an ancestor of node 2.
      return [ node1 ];
    }

    if (ancestors2.length == i + 1)
    {
      // Node 2 is an ancestor of node 1.
      return [ node2 ];
    }

    ++i;
  }
}

/** Returns all the parents of a node, as a list.
 *
 * The first item in the list will be the body.
 * The last item in the list will be node.
 */
export function ancestorList(node: Node) : Array<Node>
{
  // Store the body in a local variable to avoid lots of global lookups.
  let theBody = document.body;
  let currNode = node;

  let theList: Array<Node> = [];

  while (true)
  {
    theList.push(currNode);
    if (currNode == theBody)
    {
      break;
    }
    currNode = currNode.parentNode;
  }

  theList.reverse();
  return theList;
}

/** Selects all nodes that are immediate children only.
 *
 * @param rootNode The DOM node whose children will be searched.
 * @param selector A selector string.
 *
 * @return An array of nodes.
 */
export function selectChildren(rootNode: Element,
                               selector: string) : Array<Node>
{
  let nodes = rootNode.querySelectorAll(selector);

  let ret: Array<Node> = [];
  for (let i = 0; i < nodes.length; ++i)
  {
    if (nodes[i].parentNode == rootNode)
    {
      ret.push(nodes[i]);
    }
  }

  return ret;
}

/** Reads a directive's attribute, thowing an exception if it doesn't exist.
 *
 * @param attrs The Angular attributes object.
 * @param name The name of the attribute to read.
 * @param attrType The type of the attribute being read.  Can be 'string',
 *                 'int', 'float', or 'bool'.
 * @param directiveName The name of the directive being processed.
 * @param defaultValue [Optional] If provided, this is the value that
 *                     will be returned when no value has been
 *                     included in the attributes.  If this is not
 *                     provided, a missing value will result in an exception.
 *
 * @return The value of the attribute.
 */
  /*
  export function getAttr<T>(attrs: angular.IAttributes,
                             name: string,
                             attrType: string,
                             directiveName: string,
                             defaultValue?: T) : T
{
  // The name of the attribute as it appears in the XML/HTML that the user
  // types.
  let xmlName = axeString.camelCaseToDashes(name);

  if (typeof attrs[name] == 'undefined' ||
      (attrs[name] === '' && attrType != 'string'))
  {
    if (typeof defaultValue != 'undefined')
    {
      return defaultValue;
    }
    else
    {
      let msg = 'Error processing ' + directiveName + ' directive. ' +
        'The ' + xmlName + ' attribute is missing.';
      throw new Error(msg);
    }
  }

  let valueStr: string = attrs[name];
  let value: T = null;

  if (attrType == 'int')
  {
    let intValue: number = parseInt(valueStr);
    if (isNaN(intValue))
    {
      let msg = 'Error processing ' + directiveName + ' directive. ' +
        'The ' + xmlName + ' attribute is not a valid int.  Value: ' + valueStr;
      throw new Error(msg);
    }
    value = <T><any>intValue;
  }
  else if (attrType == 'float')
  {
    let floatValue: number = parseFloat(valueStr);
    if (isNaN(floatValue))
    {
      let msg = 'Error processing ' + directiveName + ' directive. ' +
        'The ' + xmlName + ' attribute is not a valid float. ' +
        ' Value: ' + valueStr;
      throw new Error(msg);
    }
    value = <T><any>floatValue;
  }
  else if (attrType == 'bool')
  {
    if (valueStr != 'true' && valueStr != 'false')
    {
      let msg = 'Error processing ' + directiveName + ' directive. ' +
        'The ' + xmlName + ' attribute is not a valid bool.  ' +
        'Value: ' + valueStr;
      throw new Error(msg);
    }
    value = <T><any>(valueStr == 'true');
  }
  else if (attrType == 'string')
  {
    value = <T><any>valueStr;
  }
  else
  {
    let msg = 'Unknown attrType: ' + attrType;
    throw new Error(msg);
  }

  return value;
};
  */

/** Create a getter and setter for a property that forwards to _data.
 */
  /* TODO delete
export function defineAccessor(target: any, key: string, setter: any) : void
{
  // specifically false, this should not also match undefined.
  if (setter === false)
  {
    setter = undefined;
  }
  else if (typeof setter != "function")
  {
    setter = function(value: any) : void
    {
      let save = this._data[key] != value;
      this._data[key] = value;
      if (save)
      {
        this.save();
      }
    };
  }
  function getter() : any
  {
    return this._data[key];
  }
  Object.defineProperty(target, key, {
    "get": getter,
    "set": setter,
    "enumerable": true
  });
};
*/

/** Create a read only property, optionally hidden from enumerability.
 */
/* TODO delete
export function defineReadOnly(target: any, key: string, value: any,
                               hidden: boolean) : void
{
  Object.defineProperty(target, key, {
    "value": value,
    "enumerable": !hidden,
  });
};

export function copy<T>(targetObj: T) : T
{
  return JSON.parse(JSON.stringify(targetObj));
};
*/

} // end namespace axe.data
