//------------------------------------------------------------------
//  date.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

//------------------------------------------------------------------
import { Day } from '../date/Day';
//------------------------------------------------------------------


//------------------------------------------------------------------
// Constants
//------------------------------------------------------------------

/** The number of milliseconds in a day.
 */
export const msecPerDay: number = 24 * 60 * 60 * 1000;

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Returns true if a datetime falls within the specified day.
 *
 * @param datetime A Date object to test.
 * @param day A Date object to see if the datetime falls within.
 */
export function datetimeWithinDay(datetime: Date, day: Date) : boolean
{
  let msec = datetime.getTime();
  let beginMsec = day.getTime();
  let endMsec = beginMsec + msecPerDay;

  return msec >= beginMsec && msec < endMsec;
}

/** Returns a Date object that is the current date.
 */
export function today() : Date
{
  let todayValue = new Date();
  discardDayFraction(todayValue);

  return todayValue;
}

/** Returns the date portion of a datetime.
 *
 * @param datetime A Date object to be converted to date-only Date.
 *
 * @return A new Date object that has any fraction of a day discarded.
 */
export function floorDate(datetime: Date) : Date
{
  let dateOnly = new Date(datetime.getTime());
  discardDayFraction(dateOnly);

  return dateOnly;
}

/** Returns the date portion of a datetime set to the next Date
 *  (if there are fractional days).
 * This acts like Math.ceil().
 *
 * @param datetime A Date object to be converted to date-only Date.
 *
 * @return A new Date object that has any fraction of a day discarded.
 */
export function ceilDate(datetime: Date) : Date
{
  let newDate: Date = new Date(datetime.getTime());
  if (newDate.getHours() > 0 || newDate.getMinutes() > 0 ||
      newDate.getSeconds() > 0 || newDate.getMilliseconds() > 0)
  {
    addDays(newDate, 1);
  }

  discardDayFraction(newDate);

  return newDate;
}

/** Returns the date portion of a datetime set to the closest Date().
 * This acts like Math.round().
 *
 * @param datetime A Date object to be converted to date-only Date.
 *
 * @return A new Date object that has any fraction of a day discarded.
 */
export function roundDate(datetime: Date) : Date
{
  let newDate = new Date(datetime.getTime());
  if (newDate.getHours() >= 12)
  {
    newDate = addDays(newDate, 1);
  }

  discardDayFraction(newDate);

  return newDate;
}

/** Makes the Date object hold a date only, discarding any time info.
 *
 * @param datetime A Date object to be converted to date-only Date.
 */
export function discardDayFraction(datetime: Date) : void
{
  datetime.setHours(0);
  datetime.setMinutes(0);
  datetime.setSeconds(0);
  datetime.setMilliseconds(0);
}

/** Adds a number of days to a date.
 *
 * @param date The date to add to.
 * @param numDays The number of days to add to the date.
 *
 * @return A newly created date object.
 */
export function addDays(date: Date, numDays: number) : Date
{
  if (numDays != Math.round(numDays))
  {
    throw Error("Error: addDays() does not support fractions of days.");
  }

  let newDate = new Date(date.valueOf());
  newDate.setDate(newDate.getDate() + numDays);
  return newDate;
}

/** Adds an interger number of months to a date.
 *
 * @param date The date to add to.
 * @param numMonths The number of months to add to the date.
 *
 * @return A newly created date object.
 */
export function addMonths(date: Date, numMonths: number) : Date
{
  if (numMonths != Math.round(numMonths))
  {
    throw Error("Error: addMonths() does not support fractions of months.");
  }

  let newDate = new Date(date.valueOf());
  newDate.setMonth(newDate.getMonth() + numMonths);
  return newDate;
}

/** Adds a number of hours to a date.
 *
 * This function does not really add the number of hours.
 * It adds numHours * 60 * 60 * 1000 millseconds, ignoring
 * leap seconds.
 *
 * @param date The date to add to.
 * @param numHours The number of hours to add to the date.
 *
 * @return A newly created date object.
 */
export function addHours(date: Date, numHours: number) : Date
{
  let millisecPerHour = 1000 * 60 * 60;
  let newTime = date.getTime();
  newTime += millisecPerHour * numHours;
  let newDate = new Date(Math.round(newTime));
  return newDate;
}

/** Determines the maximum of two dates.
 *  This will return the first date object when they are equivalent.
 *
 * @param date The first date to compare with.
 * @param otherDate The second date to compare with.
 *
 * @return The date the occurs latest.
 */
export function max(date: Date, otherDate: Date) : Date
{
  if (date >= otherDate)
  {
    return date;
  }
  return otherDate;
}

/** Determines the minimum of two dates.
 *  This will return the first date object when they are equivalent.
 *
 * @param date The first date to compare with.
 * @param otherDate The second date to compare with.
 *
 * @return The date that occurs earliest.
 */
export function min(date: Date, otherDate: Date) : Date
{
  if (date <= otherDate)
  {
    return date;
  }
  return otherDate;
}

/** Parses a string of the format 'YYYY-MM-DD' and returns a Day.
 */
export function parseDate(dateStr: string) : Day
{
  if (dateStr == "") {
    return null;
  }

  let expectedFmt = 'YYYY-MM-DD';

  if (dateStr.length != expectedFmt.length) {
    let msg = "Failed to parse date '" + dateStr + "'.  " +
      "Expected format: " + expectedFmt;
    throw new Error(msg);
  }

  let dateElems = dateStr.split('-');

  let year = parseInt(dateElems[0]);
  let month = parseInt(dateElems[1]) - 1;
  let dom = parseInt(dateElems[2]);

  let dateObj = new Date(year, month, dom);
  let dayObj = new Day(dateObj.getTime());

  return dayObj;
}

/** Returns a string with format 'yyyy-mm-dd' for the given Date.
 *
 * @param dateObj A Date object to format into a string.
 *
 * @return The date string.
 */
export function formatDate(day: Day) : string
{
  if (!day) {
    return "";
  }

  let dateObj: Date = day.asDate();

  let year = "" + dateObj.getFullYear();
  let month = "" + (dateObj.getMonth() + 1);
  let dom = "" + dateObj.getDate();

  // Pad with zeros if necessary.

  if (month.length < 2) {
    month = "0" + month;
  }
  if (dom.length < 2) {
    dom = "0" + dom;
  }

  let dateStr = year + "-" + month + "-" + dom;
  return dateStr;
}

/** Parses a string with format 'YYYY-MM-DDThh24:mm:ss.sssZ' and returns a Date.
 */
export function parseDateTime(dateStr: string) : Date
{
  if (!dateStr)
  {
    return null;
  }

  let expectedFmt = 'YYYY-MM-DDThh:mm:ss.sssZ';

  if (dateStr.length != expectedFmt.length)
  {
    let msg = "Failed to parse date time '" + dateStr + "'.  " +
      "Expected format: " + expectedFmt;
    throw new Error(msg);
  }

  let yearStr = dateStr.substring(0, 4);
  let monthStr = dateStr.substring(5, 7);
  let domStr = dateStr.substring(8, 10);
  let hoursStr = dateStr.substring(11, 13);
  let minsStr = dateStr.substring(14, 16);
  let secsStr = dateStr.substring(17, 19);
  let msecsStr = dateStr.substring(20, 23);

  let year = parseInt(yearStr);
  let month = parseInt(monthStr) - 1;
  let dom = parseInt(domStr);
  let hours = parseInt(hoursStr);
  let mins = parseInt(minsStr);
  let secs = parseInt(secsStr);
  let msecs = parseInt(msecsStr);

  let d = new Date(year, month, dom, hours, mins, secs, msecs);
  d.setUTCFullYear(year);
  d.setUTCMonth(month);
  d.setUTCDate(dom);
  d.setUTCHours(hours);
  d.setUTCMinutes(mins);
  d.setUTCSeconds(secs);
  d.setUTCMilliseconds(msecs);
  return d;
}

/** Returns a string with format 'yyyy-mm-ddThh:mm:ss.sssZ' for the given Date.
 *
 * @param dateObj A Date object to format into a string.
 *
 * @return The date string.
 */
export function formatDateTime(dateObj?: Date) : string
{
  if (!dateObj)
  {
    return "";
  }

  let year = "" + dateObj.getUTCFullYear();
  let month = "" + (dateObj.getUTCMonth() + 1);
  let dom = "" + dateObj.getUTCDate();
  let hours = "" + dateObj.getUTCHours();
  let mins = "" + dateObj.getUTCMinutes();
  let secs = "" + dateObj.getUTCSeconds();
  let msecs = "" + dateObj.getUTCMilliseconds();

  // Pad with zeros if necessary.

  if (month.length < 2)
  {
    month = "0" + month;
  }
  if (dom.length < 2)
  {
    dom = "0" + dom;
  }
  if (hours.length < 2)
  {
    hours = "0" + hours;
  }
  if (mins.length < 2)
  {
    mins = "0" + mins;
  }
  if (secs.length < 2)
  {
    secs = "0" + secs;
  }
  while (msecs.length < 3)
  {
    msecs = "0" + msecs;
  }

  let dateStr = year + "-" + month + "-" + dom + "T" + hours + ":" + mins +
    ":" + secs + "." + msecs + "Z";
  return dateStr;
}

/** Returns a string of the format 'MM/DD/YYYY' for the given Date.
 *
 * @param d A Date object to format into a string.
 * @return The date string.
 */
export function formatPrettyDate(dateObj?: Date) : string
{
  if (!dateObj)
  {
    return "";
  }

  let dateStr = "" + (dateObj.getMonth() + 1) + "/" + dateObj.getDate() + "/" +
    dateObj.getFullYear();
  return dateStr;
}

/** Returns a string of the format 'MMM DD' for the given Date.
 *
 * @param d A Date object to format into a string.
 * @return The date string.
 */
export function formatShortDate(dateObj: Date) : string
{
  if (!dateObj)
  {
    return "";
  }

  let monthAbbr = [ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" ];
  let dateStr = monthAbbr[(dateObj.getMonth())] + " " + dateObj.getDate();
  return dateStr;
}

/** Returns a string of the format 'MM/DD/YYYY hh:mm AM' for the given Date.
 *
 * @param d A Date object to format into a string.
 * @return The date string.
 */
export function formatPrettyDateTime(dateObj: Date) : string
{
  if (!dateObj)
  {
    return "";
  }

  let hours12 = dateObj.getHours() % 12;
  if (hours12 == 0) {
    hours12 = 12;
  }

  let mm = dateObj.getMinutes().toString();
  while (mm.length < 2)
  {
    mm = "0" + mm;
  }

  let amPm = ((dateObj.getHours() < 12) ? "AM" : "PM");
  let dateStr = "" + (dateObj.getMonth() + 1) + "/" + dateObj.getDate() + "/" +
                dateObj.getFullYear() + " " +  hours12 + ":" +
                mm + " " + amPm;
  return dateStr;
}

/** Returns a time string of the format 'hh:mm:ss' for the given Date.
 *
 * @param d A Date object to format into a string.
 * @return The time string.
 */
export function formatTimeSecs(dateObj: Date) : string
{
  let hh = dateObj.getHours().toString();

  let mm = dateObj.getMinutes().toString();
  while (mm.length < 2)
  {
    mm = "0" + mm;
  }
  let ss = dateObj.getSeconds().toString();
  while (ss.length < 2)
  {
    ss = "0" + ss;
  }

  let timeStr = hh + ":" + mm + ":" + ss;
  return timeStr;
}

/** Returns a time string of the format 'hh:mm:ss.ssss' for the given Date.
 *
 * @param d A Date object to format into a string.
 * @return The time string.
 */
export function formatTimeMillis(dateObj: Date) : string
{
  let hh = dateObj.getHours().toString();

  let mm = dateObj.getMinutes().toString();
  while (mm.length < 2)
  {
    mm = "0" + mm;
  }
  let ss = dateObj.getSeconds().toString();
  while (ss.length < 2)
  {
    ss = "0" + ss;
  }
  let millis = dateObj.getMilliseconds().toString();
  while (millis.length < 2)
  {
    millis = "0" + millis;
  }

  let timeStr = hh + ":" + mm + ":" + ss + "." + millis;
  return timeStr;
}

/** Formats an interval with a string like '3 hours ago'.
 *
 * The returned string will be one of:
 *
 *   0/15/30/45 minutes ago
 *   x hours ago
 *   x days ago
 *
 * @param milliseconds The number of milliseconds that is the interval.
 *
 * @return A string describing how much earlier the number of milliseconds is.
 */
export function formatAgo(milliseconds: number) : string
{
  let numDays = milliseconds / (1000 * 60 * 60 * 24);
  let numHours = milliseconds / (1000 * 60 * 60);
  let numQuarterHours = milliseconds / (1000 * 60 * 15);

  numDays = Math.floor(numDays);
  numHours = Math.floor(numHours);
  numQuarterHours = Math.floor(numQuarterHours);

  if (numDays > 1)
  {
    return numDays + " days ago";
  }
  else if (numDays > 0)
  {
    return "1 day ago";
  }
  else if (numHours > 1)
  {
    return numHours + " hours ago";
  }
  else if (numHours > 0)
  {
    return "1 hour ago";
  }
  else if (numQuarterHours > 0)
  {
    return (numQuarterHours * 15) + " minutes ago";
  }
  else
  {
    return "less than 15 minutes ago";
  }
}

/** Compare two dates so they will sort in ascending order.
 */
export function compareAsc(date1: Date, date2: Date) : number
{
  return date1.getTime() - date2.getTime();
}


/** Compare two dates so they will sort in descending order.
 */
export function compareDesc(date1: Date, date2: Date) : number
{
  return date2.getTime() - date1.getTime();
}
