//------------------------------------------------------------------
//  TestAngle.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../util/string";
import { UnitTest } from "../../unittest/UnitTest";
import { UnitTestRunner } from "../../unittest/UnitTestRunner";
import { Angle } from "../Angle";
//------------------------------------------------------------------

/** Unit test for the Angle class.
 */
export class TestAngle extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new Angle object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /* Test basic angle functionality.
   */
  testAngleBasic() : void
  {
    let angle: Angle = Angle.degrees(0);
    this.assertClose('degrees -> radians', 0, angle.radians);

    angle = Angle.radians(0);
    this.assertClose('radians -> degrees', 0, angle.degrees);

    angle = Angle.degrees(180);
    this.assertClose('degrees -> radians', Math.PI, angle.radians);

    angle = Angle.radians(Math.PI);
    this.assertClose('radians -> degrees', 180, angle.degrees);

    angle = Angle.degrees(360);
    this.assertClose('normalized degrees', 0, angle.degrees);
    this.assertClose('normalized radians', 0, angle.radians);

    angle = Angle.radians(2 * Math.PI);
    this.assertClose('normalized degrees', 0, angle.degrees);
    this.assertClose('normalized radians', 0, angle.radians);

    angle = Angle.degrees(90).plus(Angle.degrees(15));
    this.assertClose('plus', 105, angle.degrees);

    angle = Angle.radians(Math.PI).minus(Angle.degrees(45));
    this.assertClose('minus', 135, angle.degrees);

    angle = Angle.degrees(90).times(2);
    this.assertClose('times', 180, angle.degrees);

    angle = Angle.degrees(90).dividedBy(2);
    this.assertClose('dividedBy', 45, angle.degrees);

    this.assertClose('cos 0', 1, Angle.degrees(0).cos());
    this.assertClose('cos 180', -1, Angle.degrees(180).cos());
    this.assertClose('cos 270', 0, Angle.degrees(270).cos());

    this.assertClose('sin 0', 0, Angle.degrees(0).sin());
    this.assertClose('sin 90', 1, Angle.degrees(90).sin());
    this.assertClose('sin 180', 0, Angle.degrees(180).sin());
    this.assertClose('cos 270', -1, Angle.degrees(270).sin());

    this.assertClose('m2m 0', 90, Angle.degrees(0).mathToMapAngle().degrees);
    this.assertClose('m2m 90', 0, Angle.degrees(90).mapToMathAngle().degrees);
  }

  /** Tests converting to/from JSON.
   */
  testAngleJson() : void
  {
    interface JsonifiedAngle
    {
      _class: string;
      degrees: number;
    }

    let jsonObj: JsonifiedAngle = {
      '_class': 'Angle',
      'degrees': 90
    };

    let angle: Angle = Angle.fromJson(jsonObj);
    this.assertClose('javascript object', 90, angle.degrees);

    jsonObj = <JsonifiedAngle>angle.toJson();
    this.assertClose('JSON object', 90, jsonObj.degrees);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestAngle

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestAngle());
